<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * RCCC Homeowner shortcodes (registration, login, logout, account)
 * Safe redirect to Get Matched page (uses page_id=11).
 */

/**
 * [homeowner_register]
 */
function rccc_home_register_shortcode( $atts ) {
	if ( is_user_logged_in() ) {
		return '<p>' . esc_html__( 'You are already logged in.', 'rccc-homeowner' ) . '</p>';
	}

	$output = '';

	// Handle form submission
	if ( isset( $_POST['rccc_register_nonce'] ) && wp_verify_nonce( wp_unslash( $_POST['rccc_register_nonce'] ), 'rccc_register_action' ) ) {

		$username = sanitize_user( wp_unslash( $_POST['rccc_username'] ?? '' ) );
		$email    = sanitize_email( wp_unslash( $_POST['rccc_email'] ?? '' ) );
		$pass     = $_POST['rccc_password'] ?? '';
		$pass2    = $_POST['rccc_password2'] ?? '';
		$display  = sanitize_text_field( wp_unslash( $_POST['rccc_display_name'] ?? '' ) );

		$errors = new WP_Error();

		// Basic validation
		if ( empty( $username ) || empty( $email ) || empty( $pass ) ) {
			$errors->add( 'required', __( 'Please fill required fields.', 'rccc-homeowner' ) );
		}
		if ( $pass !== $pass2 ) {
			$errors->add( 'pass_mismatch', __( 'Passwords do not match.', 'rccc-homeowner' ) );
		}
		if ( username_exists( $username ) ) {
			$errors->add( 'user_exists', __( 'Username already taken.', 'rccc-homeowner' ) );
		}
		if ( email_exists( $email ) ) {
			$errors->add( 'email_exists', __( 'Email already registered.', 'rccc-homeowner' ) );
		}

		if ( ! $errors->has_errors() ) {
			$user_id = wp_create_user( $username, $pass, $email );
			if ( is_wp_error( $user_id ) ) {
				$errors->add( 'create_failed', $user_id->get_error_message() );
			} else {
				// Set role and display name
				$user = new WP_User( $user_id );
				$user->set_role( 'homeowner' );
				if ( ! empty( $display ) ) {
					wp_update_user( array( 'ID' => $user_id, 'display_name' => $display ) );
				}

				// Log in the user
				wp_set_current_user( $user_id );
				wp_set_auth_cookie( $user_id );

				// Build params to send to Get Matched page (name + email)
				$name_param = ! empty( $display ) ? $display : $username;
				$params = array(
					'name'  => $name_param,
					'email' => $email,
				);

				// Use page_id fallback so permalinks/rewrite don't cause 404s
				$base = site_url( '/?page_id=11' );
				$redirect_url = add_query_arg( $params, $base );

				// Optional logging (will appear in PHP/WordPress logs if enabled)
				if ( function_exists( 'error_log' ) ) {
					error_log( 'rccc_register: redirecting new user_id=' . intval( $user_id ) . ' to ' . $redirect_url );
				}

				// Server-side redirect if possible
				$safe = esc_url_raw( $redirect_url );
				if ( ! headers_sent() ) {
					wp_safe_redirect( $safe );
					exit;
				}

				// Fallback client-side redirect
				?>
				<script type="text/javascript">
				(function(){
					var url = <?php echo json_encode( $safe ); ?>;
					try {
						window.location.replace(url);
					} catch(e) {
						window.location.href = url;
					}
				})();
				</script>
				<noscript>
					<meta http-equiv="refresh" content="0;url=<?php echo esc_attr( $safe ); ?>" />
				</noscript>
				<?php
				exit;
			}
		}

		// Display any validation errors
		if ( $errors->has_errors() ) {
			foreach ( $errors->get_error_messages() as $msg ) {
				$output .= '<div class="rccc-error" style="color:#a00;margin:8px 0;">' . esc_html( $msg ) . '</div>';
			}
		}
	}

	// Render registration form (no category/zip)
	$output .= '<form method="post" class="rccc-home-register" autocomplete="off">';
	$output .= wp_nonce_field( 'rccc_register_action', 'rccc_register_nonce', true, false );
	$output .= '<p><label>' . esc_html__( 'Username', 'rccc-homeowner' ) . '<br><input name="rccc_username" required></label></p>';
	$output .= '<p><label>' . esc_html__( 'Email', 'rccc-homeowner' ) . '<br><input name="rccc_email" type="email" required></label></p>';
	$output .= '<p><label>' . esc_html__( 'Password', 'rccc-homeowner' ) . '<br><input name="rccc_password" type="password" required></label></p>';
	$output .= '<p><label>' . esc_html__( 'Confirm Password', 'rccc-homeowner' ) . '<br><input name="rccc_password2" type="password" required></label></p>';
	$output .= '<p><label>' . esc_html__( 'Display Name', 'rccc-homeowner' ) . '<br><input name="rccc_display_name"></label></p>';
	$output .= '<input type="hidden" name="redirect_to" value="' . esc_attr( site_url( '/?page_id=11' ) ) . '">';
	$output .= '<p><button type="submit" class="rccc-btn">' . esc_html__( 'Create Account', 'rccc-homeowner' ) . '</button></p>';
	$output .= '</form>';

	return $output;
}
add_shortcode( 'homeowner_register', 'rccc_home_register_shortcode' );

/**
 * [homeowner_login]
 */
function rccc_home_login_shortcode( $atts ) {
	if ( is_user_logged_in() ) {
		return '<p>' . esc_html__( 'You are already logged in.', 'rccc-homeowner' ) . '</p>';
	}

	$output = '';

	if ( isset( $_POST['rccc_login_nonce'] ) && wp_verify_nonce( wp_unslash( $_POST['rccc_login_nonce'] ), 'rccc_login_action' ) ) {
		$creds = array();
		$creds['user_login']    = sanitize_text_field( wp_unslash( $_POST['rccc_user'] ?? '' ) );
		$creds['user_password'] = $_POST['rccc_pass'] ?? '';
		$creds['remember']      = isset( $_POST['rccc_remember'] );

		$user = wp_signon( $creds, is_ssl() );

		if ( is_wp_error( $user ) ) {
			$output .= '<div class="rccc-error" style="color:#a00;margin:8px 0;">' . esc_html( $user->get_error_message() ) . '</div>';
		} else {
			$raw_redirect  = wp_unslash( $_POST['redirect_to'] ?? '' );
			$safe_redirect = wp_validate_redirect( $raw_redirect, home_url( '/homeowner-account/' ) );
			$safe_redirect = esc_url_raw( $safe_redirect );
			wp_safe_redirect( $safe_redirect );
			exit;
		}
	}

	$output .= '<form method="post" class="rccc-home-login">';
	$output .= wp_nonce_field( 'rccc_login_action', 'rccc_login_nonce', true, false );
	$output .= '<p><label>' . esc_html__( 'Username or Email', 'rccc-homeowner' ) . '<br><input name="rccc_user" required></label></p>';
	$output .= '<p><label>' . esc_html__( 'Password', 'rccc-homeowner' ) . '<br><input name="rccc_pass" type="password" required></label></p>';
	$output .= '<p><label><input name="rccc_remember" type="checkbox"> ' . esc_html__( 'Remember Me', 'rccc-homeowner' ) . '</label></p>';
	$output .= '<input type="hidden" name="redirect_to" value="' . esc_attr( home_url( '/homeowner-account/' ) ) . '">';
	$output .= '<p><button type="submit" class="rccc-btn">' . esc_html__( 'Login', 'rccc-homeowner' ) . '</button></p>';
	$output .= '</form>';

	return $output;
}
add_shortcode( 'homeowner_login', 'rccc_home_login_shortcode' );

/**
 * [homeowner_logout]
 */
function rccc_home_logout_shortcode( $atts ) {
	if ( ! is_user_logged_in() ) {
		return '';
	}
	$redirect = ! empty( $atts['redirect'] ) ? esc_url_raw( $atts['redirect'] ) : home_url();
	$url = wp_logout_url( $redirect );
	return '<a class="rccc-logout" href="' . esc_url( $url ) . '">' . esc_html__( 'Log out', 'rccc-homeowner' ) . '</a>';
}
add_shortcode( 'homeowner_logout', 'rccc_home_logout_shortcode' );

/**
 * [homeowner_account]
 * Minimal account dashboard to avoid extra surface area during fixes.
 */
function rccc_home_account_shortcode( $atts ) {
	if ( ! is_user_logged_in() ) {
		return '<p>' . esc_html__( 'Please log in to view your account.', 'rccc-homeowner' ) . '</p>';
	}

	$user_id = get_current_user_id();
	$output  = '<div class="rccc-account">';
	$output .= '<h3>' . esc_html__( 'Your Account', 'rccc-homeowner' ) . '</h3>';
	$output .= '<p>' . sprintf( esc_html__( 'Hello %s', 'rccc-homeowner' ), esc_html( wp_get_current_user()->display_name ?: wp_get_current_user()->user_login ) ) . '</p>';
	$output .= '<p><a class="rccc-btn" href="' . esc_url( home_url( '/homeowner-account/' ) ) . '">' . esc_html__( 'Manage Account', 'rccc-homeowner' ) . '</a></p>';
	$output .= do_shortcode( '[homeowner_logout]' );
	$output .= '</div>';

	return $output;
}
add_shortcode( 'homeowner_account', 'rccc_home_account_shortcode' );