<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WPForms -> RCCC integration (debug-enabled)
 * Replaces the previous integration temporarily to log what values were seen server-side.
 * Keep this only for testing; you can remove the debug logs afterward.
 */

add_action( 'wpforms_process_complete', 'rccc_wpforms_process_complete', 10, 4 );

function rccc_wpforms_process_complete( $fields, $entry, $form_data, $entry_id ) {
	$target_form_id = 27;
	$form_id = intval( $form_data['id'] ?? 0 );

	// Log entry even when not target to help debugging
	if ( function_exists( 'error_log' ) ) {
		error_log( 'rccc_wpforms_integration: process_complete hook called form_id=' . intval( $form_id ) . ' entry_id=' . intval( $entry_id ) );
		error_log( 'rccc_wpforms_integration: raw fields: ' . print_r( $fields, true ) );
	}

	if ( $form_id !== $target_form_id ) {
		return;
	}

	// Only act if a logged-in user submitted
	$user_id = get_current_user_id();
	if ( function_exists( 'error_log' ) ) {
		error_log( 'rccc_wpforms_integration: form=' . $form_id . ' entry=' . $entry_id . ' user_id=' . intval( $user_id ) );
	}

	if ( ! $user_id ) {
		// Not logged in — log and exit
		if ( function_exists( 'error_log' ) ) {
			error_log( 'rccc_wpforms_integration: skipping because user_id=0' );
		}
		return;
	}

	// Mapping (your field IDs)
	$mapping = array(
		'name'     => '1',
		'email'    => '2',
		'phone'    => '3',
		'address'  => '6',
		'category' => '7',
		'date'     => '',
	);

	$config = array(
		'project_cpt' => 'rccc_project',
		'lead_cpt'    => 'project',
	);

	// Helper to read field values robustly (same as before)
	$get_field_value = function( $id ) use ( $fields ) {
		$id = (string) $id;
		if ( isset( $fields[ $id ] ) && array_key_exists( 'value', $fields[ $id ] ) ) {
			$val = $fields[ $id ]['value'];
			if ( is_array( $val ) ) {
				$parts = array();
				foreach ( $val as $v ) {
					if ( is_array( $v ) ) {
						$parts[] = implode( ' ', array_filter( array_map( 'strval', $v ) ) );
					} else {
						$parts[] = (string) $v;
					}
				}
				return trim( implode( ', ', array_filter( array_map( 'trim', $parts ) ) ) );
			} elseif ( is_object( $val ) ) {
				return trim( wp_json_encode( $val ) );
			}
			return trim( (string) $val );
		}
		// fallback
		foreach ( $fields as $f ) {
			if ( isset( $f['value'] ) && is_scalar( $f['value'] ) && strlen( trim( (string) $f['value'] ) ) ) {
				return trim( (string) $f['value'] );
			}
		}
		return '';
	};

	// Read values
	$name_raw     = $get_field_value( $mapping['name'] ?? '' );
	$email_raw    = $get_field_value( $mapping['email'] ?? '' );
	$phone_raw    = $get_field_value( $mapping['phone'] ?? '' );
	$address_raw  = $get_field_value( $mapping['address'] ?? '' );
	$category_raw = $get_field_value( $mapping['category'] ?? '' );

	// Sanitize
	$name     = sanitize_text_field( $name_raw );
	$email    = sanitize_email( $email_raw );
	$phone    = sanitize_text_field( $phone_raw );
	$address  = sanitize_textarea_field( $address_raw );
	$category = sanitize_text_field( $category_raw );

	// Log extracted values
	if ( function_exists( 'error_log' ) ) {
		error_log( "rccc_wpforms_integration: extracted name={$name} email={$email} phone={$phone} address={$address} category={$category}" );
	}

	// If any of these are empty but WPForms entry shows them, that indicates mapping mismatch — paste the raw fields above.
	// Continue with the same project/lead creation logic as previously (omitted here for brevity)
	// For now just attempt to update usermeta and create the posts if name or category present.

	if ( $name ) {
		wp_update_user( array( 'ID' => $user_id, 'display_name' => $name ) );
		$parts = preg_split( '/\s+/', $name );
		if ( count( $parts ) === 1 ) {
			update_user_meta( $user_id, 'first_name', $parts[0] );
		} else {
			update_user_meta( $user_id, 'first_name', array_shift( $parts ) );
			update_user_meta( $user_id, 'last_name', implode( ' ', $parts ) );
		}
	}
	if ( $email ) update_user_meta( $user_id, 'rccc_contact_email', $email );
	if ( $phone ) update_user_meta( $user_id, 'phone', $phone );
	if ( $address ) update_user_meta( $user_id, 'address', $address );
	if ( $category ) update_user_meta( $user_id, 'preferred_service', $category );

	// Create minimal project and lead posts (so we can confirm creation)
	$proj_title = $name ? sprintf( '%s — Project', $name ) : sprintf( 'Project %s', date_i18n( 'Y-m-d H:i:s' ) );
	$postarr = array(
		'post_type'    => $config['project_cpt'],
		'post_title'   => wp_strip_all_tags( $proj_title ),
		'post_content' => trim( "Phone: $phone\nAddress: $address\nCategory: $category\nEmail: $email" ),
		'post_status'  => 'publish',
		'post_author'  => $user_id,
	);
	$project_id = wp_insert_post( $postarr );
	if ( is_wp_error( $project_id ) || ! $project_id ) {
		if ( function_exists( 'error_log' ) ) {
			error_log( 'rccc_wpforms_integration: failed to create project for entry ' . intval( $entry_id ) . ' error: ' . ( is_wp_error( $project_id ) ? $project_id->get_error_message() : 'unknown' ) );
		}
	} else {
		update_post_meta( $project_id, '_rccc_form_entry_id', intval( $entry_id ) );
		if ( function_exists( 'error_log' ) ) {
			error_log( 'rccc_wpforms_integration: created debug project ' . intval( $project_id ) . ' for entry ' . intval( $entry_id ) );
		}
	}

	// Create simple lead post
	$lead_post = array(
		'post_type'    => $config['lead_cpt'],
		'post_title'   => wp_strip_all_tags( ( $category ?: 'Service' ) . ' lead for ' . ( $name ?: 'customer' ) ),
		'post_content' => trim( "Phone: $phone\nAddress: $address\nCategory: $category\nEmail: $email" ),
		'post_status'  => 'publish',
		'post_author'  => $user_id,
	);
	$lead_id = wp_insert_post( $lead_post );
	if ( is_wp_error( $lead_id ) || ! $lead_id ) {
		if ( function_exists( 'error_log' ) ) {
			error_log( 'rccc_wpforms_integration: failed to create lead for entry ' . intval( $entry_id ) . ' error: ' . ( is_wp_error( $lead_id ) ? $lead_id->get_error_message() : 'unknown' ) );
		}
	} else {
		update_post_meta( $lead_id, '_rccc_form_entry_id', intval( $entry_id ) );
		update_post_meta( $lead_id, '_rccc_project_id', intval( $project_id ) );
		if ( function_exists( 'error_log' ) ) {
			error_log( 'rccc_wpforms_integration: created debug lead ' . intval( $lead_id ) . ' for entry ' . intval( $entry_id ) );
		}
	}
}