<?php
/**
 * Plugin Name: RCCC Lead Debugger
 * Description: Admin tool to debug contractor matching for a project/lead and optionally test sending emails to matched contractors and homeowner.
 * Version: 1.0.0
 * Author: RCCC
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/* Admin page */
add_action( 'admin_menu', function() {
    add_management_page( 'RCCC Lead Debug', 'RCCC Lead Debug', 'manage_options', 'rccc-lead-debug', 'rccc_lead_debug_page' );
} );

function rccc_lead_debug_page() {
    if ( ! current_user_can( 'manage_options' ) ) wp_die( 'Permission denied' );

    $notice = '';
    $output = '';

    if ( isset( $_POST['rccc_debug_run'] ) && check_admin_referer( 'rccc_lead_debug_nonce', 'rccc_lead_debug_nonce_field' ) ) {
        $project_id = intval( $_POST['rccc_project_id'] ?? 0 );
        $send_emails = ! empty( $_POST['rccc_send_emails'] ) ? true : false;
        if ( $project_id <= 0 ) {
            $notice = '<div class="notice notice-error"><p>Enter a valid numeric Project ID.</p></div>';
        } else {
            $output = rccc_lead_debug_run( $project_id, $send_emails );
        }
    }

    echo '<div class="wrap"><h1>RCCC Lead Debug</h1>';
    echo '<p>Enter a Project ID and run the debug to see how contractors are matched for that lead (zip & category).</p>';
    echo $notice;
    echo '<form method="post">';
    wp_nonce_field( 'rccc_lead_debug_nonce', 'rccc_lead_debug_nonce_field' );
    echo '<table class="form-table"><tr><th><label for="rccc_project_id">Project ID</label></th><td><input type="number" id="rccc_project_id" name="rccc_project_id" class="regular-text" required></td></tr>';
    echo '<tr><th>Send test emails?</th><td><label><input type="checkbox" name="rccc_send_emails" value="1"> Send test emails to matched contractors and the homeowner (real emails)</label></td></tr>';
    echo '</table>';
    submit_button( 'Run Debug', 'primary', 'rccc_debug_run' );
    echo '</form>';

    if ( $output ) {
        echo '<h2>Debug output</h2>';
        echo $output;
    }

    echo '<h2>Quick tips</h2><ul><li>Check contractor meta keys: <strong>rccc_zips_csv</strong> must include the zip (98034) exactly (watch spaces/commas).</li><li>Check contractor meta <strong>rccc_contact_email</strong> exists and is valid.</li><li>Check service taxonomy <strong>service_category</strong> contains the term used by the project.</li></ul>';

    echo '</div>';
}

/* The debug runner: gathers project meta, finds matching contractors, optionally sends test emails */
function rccc_lead_debug_run( $project_id, $do_send = false ) {
    // Keys - adjust if your project meta uses different keys
    $zip_meta_key = 'rccc_zip'; // common project zip meta
    $cat_meta_key = 'rccc_category'; // common project category meta
    $project = get_post( $project_id );
    if ( ! $project ) {
        return '<div class="notice notice-error"><p>Project ID ' . intval( $project_id ) . ' not found.</p></div>';
    }

    $zip = get_post_meta( $project_id, $zip_meta_key, true );
    $category = get_post_meta( $project_id, $cat_meta_key, true );

    $out = '<div style="background:#fff;border:1px solid #eee;padding:12px;">';
    $out .= '<strong>Project ID:</strong> ' . intval( $project_id ) . '<br>';
    $out .= '<strong>Project title:</strong> ' . esc_html( $project->post_title ) . '<br>';
    $out .= '<strong>Category meta (' . esc_html( $cat_meta_key ) . '):</strong> ' . esc_html( $category ) . '<br>';
    $out .= '<strong>ZIP meta (' . esc_html( $zip_meta_key ) . '):</strong> ' . esc_html( $zip ) . '<br><hr>';

    // Normalize values
    $zip_clean = preg_replace( '/[^0-9]/', '', (string) $zip );
    $cat_name = trim( (string) $category );

    // Attempt to find taxonomy term for category (by name or slug)
    $term = null;
    if ( $cat_name !== '' ) {
        $term = get_term_by( 'name', $cat_name, 'service_category' );
        if ( ! $term ) {
            $term = get_term_by( 'slug', sanitize_title( $cat_name ), 'service_category' );
        }
        if ( $term ) {
            $out .= '<div><strong>Category term found:</strong> ID ' . intval( $term->term_id ) . ' — ' . esc_html( $term->name ) . ' (slug ' . esc_html( $term->slug ) . ')</div>';
        } else {
            $out .= '<div class="notice notice-warning"><strong>Category term NOT found for:</strong> ' . esc_html( $cat_name ) . '</div>';
        }
    } else {
        $out .= '<div class="notice notice-warning"><strong>No category provided in project meta.</strong></div>';
    }

    // Build query to find contractors: published, with zip LIKE, and (if term found) in tax query
    $args = [
        'post_type'      => 'contractor',
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'fields'         => 'ids',
        'meta_query'     => [],
    ];

    if ( $zip_clean !== '' ) {
        // Use LIKE for CSV storage; ensure no accidental partial matches by checking whole string boundaries in code below if needed
        $args['meta_query'][] = [
            'key'     => 'rccc_zips_csv',
            'value'   => $zip_clean,
            'compare' => 'LIKE',
        ];
    }

    if ( $term ) {
        $args['tax_query'] = [[
            'taxonomy' => 'service_category',
            'field'    => 'term_id',
            'terms'    => [ intval( $term->term_id ) ],
        ]];
    }

    $out .= '<div><strong>WP_Query args:</strong><pre style="white-space:pre-wrap;">' . esc_html( print_r( $args, true ) ) . '</pre></div>';

    $found = get_posts( $args );
    $count = is_array( $found ) ? count( $found ) : 0;
    $out .= '<div><strong>Contractors found:</strong> ' . intval( $count ) . '</div>';

    if ( $count === 0 ) {
        $out .= '<div class="notice notice-error"><p>No contractors matched. Common causes: ZIP stored under different meta key or not containing the zip, category term name/slug mismatch, contractor posts not published.</p></div>';
        $out .= '</div>';
        return $out;
    }

    // List each contractor with its zips and email meta
    $out .= '<ol>';
    $contractor_emails = [];
    foreach ( $found as $cid ) {
        $title = get_the_title( $cid );
        $zips = get_post_meta( $cid, 'rccc_zips_csv', true );
        $contact_email = get_post_meta( $cid, 'rccc_contact_email', true );
        $out .= '<li><strong>ID ' . intval( $cid ) . ' — ' . esc_html( $title ) . '</strong><br>';
        $out .= 'rccc_zips_csv: ' . esc_html( $zips ) . '<br>';
        $out .= 'rccc_contact_email: ' . esc_html( $contact_email ) . '<br>';
        $out .= 'Permalink: <a href="' . esc_url( get_permalink( $cid ) ) . '" target="_blank">view</a></li>';
        if ( is_email( $contact_email ) ) $contractor_emails[] = $contact_email;
    }
    $out .= '</ol>';

    // Optionally send test emails
    if ( $do_send ) {
        $out .= '<hr><h3>Sending test emails</h3>';
        // homeowner email attempt
        $homeowner_email = get_post_meta( $project_id, 'rccc_homeowner_email', true ); // adjust key if different
        $homeowner_results = '';
        if ( is_email( $homeowner_email ) ) {
            $home_sub = '[TEST] Your lead was submitted';
            $home_body = "This is a test message for Project ID {$project_id}\nCategory: {$cat_name}\nZIP: {$zip_clean}";
            $headers = [ 'From: ' . get_bloginfo( 'name' ) . ' <no-reply@' . preg_replace( '/^www\\./', '', $_SERVER['HTTP_HOST'] ) . '>' ];
            $sent = wp_mail( $homeowner_email, $home_sub, $home_body, $headers );
            $homeowner_results = $sent ? 'sent' : 'failed';
        } else {
            $homeowner_results = 'no email on project meta (key rccc_homeowner_email?)';
        }
        $out .= '<div>Homeowner email: ' . esc_html( $homeowner_results ) . '</div>';

        // contractors
        foreach ( $contractor_emails as $cemail ) {
            $sub = '[TEST] Lead match';
            $body = "This is a test notification for Project ID {$project_id}\nCategory: {$cat_name}\nZIP: {$zip_clean}";
            $headers = [ 'From: ' . get_bloginfo( 'name' ) . ' <no-reply@' . preg_replace( '/^www\\./', '', $_SERVER['HTTP_HOST'] ) . '>' ];
            $sent = wp_mail( $cemail, $sub, $body, $headers );
            $out .= '<div>Sent to ' . esc_html( $cemail ) . ': ' . ( $sent ? '<span style="color:green">OK</span>' : '<span style="color:red">FAILED</span>' ) . '</div>';
        }
    } else {
        $out .= '<p><em>Dry run only — no emails were sent. To actually send test emails check "Send test emails" above and run again.</em></p>';
    }

    $out .= '</div>';
    return $out;
}