<?php
/**
 * Plugin Name: RCCC Welcome Debugger
 * Description: Debug & repair tools to find/assign contractor posts to users so the contractor welcome page displays correctly. Provides a frontend shortcode and an admin Tools page for fixes.
 * Version: 1.0.0
 * Author: RCCC
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/* ---------- Frontend shortcode for quick checks ---------- */
/* Usage: place [rccc_welcome_debugger] on the contractor welcome page while logged-in as the contractor user. */
add_shortcode( 'rccc_welcome_debugger', function( $atts ) {
    if ( ! is_user_logged_in() ) {
        return '<div style="padding:12px;background:#fff6f6;border:1px solid #f2dede;color:#a94442;">Please log in to run the debugger (shortcode only shows for logged-in users).</div>';
    }

    $user = wp_get_current_user();
    $out = '<div style="max-width:980px;margin:12px auto;padding:14px;border:1px solid #e6eef4;border-radius:8px;background:#fff;"><h3 style="margin-top:0;color:#14532d;">RCCC Welcome Debugger (current user)</h3>';

    $out .= '<strong>User ID:</strong> ' . intval( $user->ID ) . '<br>';
    $out .= '<strong>Username:</strong> ' . esc_html( $user->user_login ) . '<br>';
    $out .= '<strong>Email:</strong> ' . esc_html( $user->user_email ) . '<br>';
    $out .= '<strong>Roles:</strong> ' . esc_html( implode( ', ', (array) $user->roles ) ) . '<br><hr>';

    // Find contractor posts authored by user
    $by_author = get_posts( [
        'post_type' => 'contractor',
        'author' => $user->ID,
        'post_status' => [ 'publish','pending','draft' ],
        'posts_per_page' => -1,
    ] );

    $out .= '<h4>Contractor posts authored by this user</h4>';
    if ( ! empty( $by_author ) ) {
        $out .= '<ul>';
        foreach ( $by_author as $p ) {
            $out .= '<li>ID ' . intval( $p->ID ) . ' — ' . esc_html( get_the_title( $p->ID ) ) . ' — <a href="' . esc_url( get_edit_post_link( $p->ID ) ) . '" target="_blank">Edit</a> — <a href="' . esc_url( get_permalink( $p->ID ) ) . '" target="_blank">View</a></li>';
        }
        $out .= '</ul>';
    } else {
        $out .= '<div style="color:#6b7280;">No contractor posts authored by this user.</div>';
    }

    // Find contractor posts with rccc_contact_email matching user's email
    $by_email = [];
    if ( is_email( $user->user_email ) ) {
        $ids = get_posts( [
            'post_type' => 'contractor',
            'post_status' => [ 'publish','pending','draft' ],
            'posts_per_page' => -1,
            'meta_query' => [
                [
                    'key' => 'rccc_contact_email',
                    'value' => $user->user_email,
                    'compare' => '=',
                ],
            ],
            'fields' => 'ids',
        ] );
        if ( ! empty( $ids ) ) {
            foreach ( $ids as $pid ) {
                $by_email[] = get_post( $pid );
            }
        }
    }

    $out .= '<h4>Contractor posts matching user email (rccc_contact_email)</h4>';
    if ( ! empty( $by_email ) ) {
        $out .= '<ul>';
        foreach ( $by_email as $p ) {
            $out .= '<li>ID ' . intval( $p->ID ) . ' — ' . esc_html( get_the_title( $p->ID ) ) . ' — Author: ' . esc_html( get_the_author_meta( 'user_login', $p->post_author ) ) . ' — <a href="' . esc_url( get_edit_post_link( $p->ID ) ) . '" target="_blank">Edit</a></li>';
        }
        $out .= '</ul>';
    } else {
        $out .= '<div style="color:#6b7280;">No contractor posts found where rccc_contact_email equals this user\'s email.</div>';
    }

    // Try to determine chosen contractor for welcome page logic (prefer by_author, then by_email, then none)
    $found = null;
    if ( ! empty( $by_author ) ) {
        $found = $by_author[0];
        $found_reason = 'author';
    } elseif ( ! empty( $by_email ) ) {
        $found = $by_email[0];
        $found_reason = 'email';
    } else {
        $found_reason = 'none';
    }

    $out .= '<hr><h4>Welcome page match</h4>';
    $out .= '<div>Match method: <strong>' . esc_html( $found_reason ) . '</strong></div>';
    if ( $found instanceof WP_Post ) {
        $logo = '';
        // try ACF field rccc_contractor_logo
        if ( function_exists( 'get_field' ) ) {
            $acf_logo = get_field( 'rccc_contractor_logo', $found->ID );
            if ( is_array( $acf_logo ) && ! empty( $acf_logo['ID'] ) ) $logo = wp_get_attachment_image_url( (int) $acf_logo['ID'], 'medium' );
            elseif ( is_numeric( $acf_logo ) ) $logo = wp_get_attachment_image_url( (int) $acf_logo, 'medium' );
            elseif ( is_string( $acf_logo ) && filter_var( $acf_logo, FILTER_VALIDATE_URL ) ) $logo = $acf_logo;
        }
        if ( ! $logo ) {
            $tid = get_post_thumbnail_id( $found->ID );
            if ( $tid ) $logo = wp_get_attachment_image_url( $tid, 'medium' );
        }

        $out .= '<div style="display:flex;gap:12px;align-items:center;margin-top:12px;">';
        if ( $logo ) {
            $out .= '<div style="width:72px;height:72px;border-radius:12px;overflow:hidden;border:1px solid #eee;"><img src="' . esc_url( $logo ) . '" style="width:100%;height:100%;object-fit:contain;"></div>';
        } else {
            $initial = get_the_title( $found->ID ) ? mb_substr( get_the_title( $found->ID ), 0, 1 ) : 'C';
            $out .= '<div style="width:72px;height:72px;border-radius:999px;background:#047857;color:#fff;display:flex;align-items:center;justify-content:center;font-weight:700;font-size:26px;">' . esc_html( mb_strtoupper( $initial ) ) . '</div>';
        }
        $out .= '<div><strong>' . esc_html( get_the_title( $found->ID ) ) . '</strong><br>';
        $out .= 'Author: ' . esc_html( get_the_author_meta( 'user_login', $found->post_author ) ) . '<br>';
        $out .= 'Service area: ' . esc_html( get_post_meta( $found->ID, 'rccc_city', true ) ) . ' · ' . esc_html( get_post_meta( $found->ID, 'rccc_zips_csv', true ) ) . '<br>';
        $out .= 'Contact email: ' . esc_html( get_post_meta( $found->ID, 'rccc_contact_email', true ) ) . '<br>';
        $out .= '<a href="' . esc_url( get_permalink( $found->ID ) ) . '" target="_blank">View public profile</a> | <a href="' . esc_url( get_edit_post_link( $found->ID ) ) . '" target="_blank">Edit</a></div>';
        $out .= '</div>';
    } else {
        $out .= '<div style="color:#b03;margin-top:8px;">No matching contractor found for the welcome page. You can ask an admin to create or assign one (admin Tools provided).</div>';
    }

    // Provide a logout link
    $out .= '<hr><div style="margin-top:8px;">' . wp_kses_post( '<a href="' . wp_logout_url( home_url( '/' ) ) . '" style="display:inline-block;padding:8px 12px;background:#047857;color:#fff;border-radius:8px;text-decoration:none;">Log out</a>' ) . '</div>';

    $out .= '</div>'; // container
    return $out;
} );

/* ---------- Admin Tools page for lookup & fixes ---------- */
add_action( 'admin_menu', function() {
    add_management_page( 'RCCC Welcome Debug', 'RCCC Welcome Debug', 'manage_options', 'rccc-welcome-debug', 'rccc_welcome_debug_admin_page' );
} );

function rccc_welcome_debug_admin_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'Permission denied' );
    }

    $msg = '';
    // handle actions: create minimal contractor, assign post to user
    if ( isset( $_POST['rccc_action'] ) && check_admin_referer( 'rccc_welcome_debug_action', 'rccc_welcome_debug_nonce' ) ) {
        $action = sanitize_text_field( $_POST['rccc_action'] );
        $user_identifier = sanitize_text_field( $_POST['rccc_user'] ?? '' );
        $target_user = null;

        if ( is_numeric( $user_identifier ) ) {
            $target_user = get_user_by( 'ID', intval( $user_identifier ) );
        } elseif ( is_email( $user_identifier ) ) {
            $target_user = get_user_by( 'email', $user_identifier );
        } else {
            $target_user = get_user_by( 'login', $user_identifier );
            if ( ! $target_user ) $target_user = get_user_by( 'slug', $user_identifier );
        }

        if ( ! $target_user ) {
            $msg = '<div class="notice notice-error"><p>User not found. Enter user ID, email or login.</p></div>';
        } else {
            if ( $action === 'create_minimal' ) {
                $company = sanitize_text_field( $_POST['rccc_company'] ?? '' );
                $created = rccc_create_minimal_contractor_for_user( $target_user->ID, $company, $target_user->user_email );
                if ( $created ) {
                    $msg = '<div class="notice notice-success"><p>Created contractor ID ' . intval( $created ) . ' for user ' . esc_html( $target_user->user_login ) . '.</p></div>';
                } else {
                    $msg = '<div class="notice notice-error"><p>Failed to create contractor post.</p></div>';
                }
            } elseif ( $action === 'assign_post' ) {
                $post_id = intval( $_POST['rccc_post_id'] ?? 0 );
                if ( $post_id && get_post( $post_id ) && get_post_type( $post_id ) === 'contractor' ) {
                    wp_update_post( [ 'ID' => $post_id, 'post_author' => $target_user->ID ] );
                    $msg = '<div class="notice notice-success"><p>Assigned contractor post ID ' . $post_id . ' to user ' . esc_html( $target_user->user_login ) . '.</p></div>';
                } else {
                    $msg = '<div class="notice notice-error"><p>Invalid contractor post ID.</p></div>';
                }
            }
        }
    }

    echo '<div class="wrap"><h1>RCCC Welcome Debug</h1>';
    echo $msg;

    echo '<form method="post" style="margin-bottom:18px;">';
    wp_nonce_field( 'rccc_welcome_debug_action', 'rccc_welcome_debug_nonce' );
    echo '<table class="form-table"><tr><th><label for="rccc_user">User (ID, email, or login)</label></th><td><input type="text" id="rccc_user" name="rccc_user" class="regular-text" placeholder="e.g. 12 or user@example.com or username" required></td></tr>';
    echo '<tr><th><label for="rccc_company">Company (optional for created contractor)</label></th><td><input type="text" id="rccc_company" name="rccc_company" class="regular-text" placeholder="Company name (optional)"></td></tr>';
    echo '</table>';
    echo '<p><button type="submit" name="rccc_action" value="create_minimal" class="button button-primary">Create Minimal Contractor for User</button> ';
    echo '<span style="margin-left:10px;color:#666;">Creates a minimal contractor CPT post authored by the selected user and sets rccc_contact_email.</span></p>';
    echo '</form>';

    echo '<hr>';

    echo '<h2>Assign existing contractor post to a user</h2>';
    echo '<form method="post">';
    wp_nonce_field( 'rccc_welcome_debug_action', 'rccc_welcome_debug_nonce' );
    echo '<table class="form-table"><tr><th>User (ID, email, or login)</th><td><input type="text" name="rccc_user" class="regular-text" required></td></tr>';
    echo '<tr><th>Contractor Post ID</th><td><input type="number" name="rccc_post_id" class="regular-text" required></td></tr></table>';
    echo '<p><button type="submit" name="rccc_action" value="assign_post" class="button">Assign Post to User</button></p>';
    echo '</form>';

    echo '<hr><h2>Quick lookup helper</h2>';
    echo '<p>To diagnose a user quickly you can place the shortcode <code>[rccc_welcome_debugger]</code> on the welcome page and open it while logged in as that user.</p>';

    echo '</div>';
}

/* Helper to create a minimal contractor for the user (returns post ID or false) */
function rccc_create_minimal_contractor_for_user( $user_id, $company = '', $contact_email = '' ) {
    $user_id = (int) $user_id;
    if ( $user_id <= 0 ) return false;

    // Double-check not to duplicate: if author already has contractor, return that
    $exists = get_posts( [
        'post_type' => 'contractor',
        'author' => $user_id,
        'posts_per_page' => 1,
        'post_status' => [ 'publish','pending','draft' ],
        'fields' => 'ids',
    ] );
    if ( ! empty( $exists ) ) {
        return (int) $exists[0];
    }

    $title = $company ? sanitize_text_field( $company ) : 'Contractor ' . $user_id;
    $pid = wp_insert_post( [
        'post_type' => 'contractor',
        'post_title' => wp_strip_all_tags( $title ),
        'post_status' => 'publish',
        'post_author' => $user_id,
        'post_content' => '',
    ], true );

    if ( is_wp_error( $pid ) || ! $pid ) return false;

    // set contact email if provided or from user
    if ( $contact_email && is_email( $contact_email ) ) {
        update_post_meta( $pid, 'rccc_contact_email', sanitize_email( $contact_email ) );
    } else {
        $u = get_user_by( 'ID', $user_id );
        if ( $u && is_email( $u->user_email ) ) {
            update_post_meta( $pid, 'rccc_contact_email', sanitize_email( $u->user_email ) );
        }
    }

    return (int) $pid;
}

/* ---------- Helpful admin notice to remind to clear caches after fixes ---------- */
add_action( 'admin_notices', function() {
    if ( ! current_user_can( 'manage_options' ) ) return;
    echo '<div class="notice notice-info"><p>RCCC Welcome Debugger active. After making changes to contractor ownership or meta, clear any page/object cache and reload the welcome page as the contractor user.</p></div>';
} );