<?php
/**
 * Plugin Name: RCCC Contractor Welcome
 * Description: Shows a personalized welcome page for contractors after they register or log in. Displays company logo, name, services and service area. Includes a simple frontend create flow for users without a contractor profile.
 * Version: 1.1.0
 * Author: RCCC
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Find a contractor post for a given user.
 * Priority:
 *  - contractor post authored by the user
 *  - contractor post with meta rccc_contact_email matching user's email
 *
 * Returns WP_Post or null.
 */
if ( ! function_exists( 'rccc_get_contractor_for_user' ) ) {
    function rccc_get_contractor_for_user( $user_id ) {
        if ( empty( $user_id ) ) {
            return null;
        }

        // 1) by author
        $by_author = get_posts( array(
            'post_type'      => 'contractor',
            'author'         => (int) $user_id,
            'post_status'    => array( 'publish', 'pending', 'draft' ),
            'posts_per_page' => 1,
            'fields'         => 'all',
        ) );
        if ( ! empty( $by_author ) ) {
            return $by_author[0];
        }

        // 2) by email meta
        $user = get_userdata( $user_id );
        if ( $user && ! empty( $user->user_email ) ) {
            $by_email = get_posts( array(
                'post_type'      => 'contractor',
                'post_status'    => array( 'publish', 'pending', 'draft' ),
                'posts_per_page' => 1,
                'meta_query'     => array(
                    array(
                        'key'     => 'rccc_contact_email',
                        'value'   => $user->user_email,
                        'compare' => '=',
                    ),
                ),
                'fields'         => 'all',
            ) );
            if ( ! empty( $by_email ) ) {
                return $by_email[0];
            }
        }

        return null;
    }
}

/**
 * Helper: get logo URL for contractor post (ACF field fallback, then featured image)
 */
if ( ! function_exists( 'rccc_get_contractor_logo_url' ) ) {
    function rccc_get_contractor_logo_url( $post_id ) {
        $logo_url = '';

        if ( function_exists( 'get_field' ) ) {
            $acf_logo = get_field( 'rccc_contractor_logo', $post_id );
            if ( is_array( $acf_logo ) && ! empty( $acf_logo['ID'] ) ) {
                $logo_url = wp_get_attachment_image_url( (int) $acf_logo['ID'], 'medium' );
            } elseif ( is_numeric( $acf_logo ) ) {
                $logo_url = wp_get_attachment_image_url( (int) $acf_logo, 'medium' );
            } elseif ( is_string( $acf_logo ) && filter_var( $acf_logo, FILTER_VALIDATE_URL ) ) {
                $logo_url = $acf_logo;
            }
        }

        if ( empty( $logo_url ) ) {
            $thumb_id = get_post_thumbnail_id( $post_id );
            if ( $thumb_id ) {
                $logo_url = wp_get_attachment_image_url( $thumb_id, 'medium' );
            }
        }

        return $logo_url;
    }
}

/**
 * Process simple frontend contractor creation form submission.
 * This is intentionally minimal: it creates a contractor post authored by current user
 * and sets a few meta fields and service terms.
 */
if ( ! function_exists( 'rccc_handle_welcome_form_submission' ) ) {
    function rccc_handle_welcome_form_submission() {
        if ( ! is_user_logged_in() ) {
            return null;
        }

        if ( empty( $_POST['rccc_create_contractor_nonce'] ) ) {
            return null;
        }

        if ( ! wp_verify_nonce( wp_unslash( $_POST['rccc_create_contractor_nonce'] ), 'rccc_create_contractor' ) ) {
            return new WP_Error( 'invalid_nonce', 'Security check failed.' );
        }

        // Basic sanitization and required fields
        $title       = isset( $_POST['rccc_company_name'] ) ? sanitize_text_field( wp_unslash( $_POST['rccc_company_name'] ) ) : '';
        $city        = isset( $_POST['rccc_city'] ) ? sanitize_text_field( wp_unslash( $_POST['rccc_city'] ) ) : '';
        $zips        = isset( $_POST['rccc_zips'] ) ? sanitize_text_field( wp_unslash( $_POST['rccc_zips'] ) ) : '';
        $services    = isset( $_POST['rccc_services'] ) ? sanitize_text_field( wp_unslash( $_POST['rccc_services'] ) ) : '';
        $contact_eml = isset( $_POST['rccc_contact_email'] ) ? sanitize_email( wp_unslash( $_POST['rccc_contact_email'] ) ) : '';

        if ( empty( $title ) ) {
            return new WP_Error( 'missing_title', 'Please provide a company name.' );
        }

        $current_user = wp_get_current_user();
        $author_id = $current_user->ID;

        $postarr = array(
            'post_title'   => $title,
            'post_type'    => 'contractor',
            'post_status'  => 'publish', // change to 'pending' if you prefer moderation
            'post_content' => '',
            'post_author'  => $author_id,
        );

        $post_id = wp_insert_post( $postarr, true );

        if ( is_wp_error( $post_id ) || ! $post_id ) {
            return new WP_Error( 'create_failed', 'Failed to create contractor profile.' );
        }

        // Save meta
        if ( $contact_eml ) {
            update_post_meta( $post_id, 'rccc_contact_email', $contact_eml );
        } else {
            // default to user email
            if ( ! empty( $current_user->user_email ) ) {
                update_post_meta( $post_id, 'rccc_contact_email', $current_user->user_email );
            }
        }

        if ( $city ) {
            update_post_meta( $post_id, 'rccc_city', $city );
        }
        if ( $zips ) {
            // normalize CSV
            $zips_norm = implode( ',', array_map( 'trim', explode( ',', $zips ) ) );
            update_post_meta( $post_id, 'rccc_zips_csv', $zips_norm );
        }

        // Services: accept comma-separated values; ensure taxonomy terms exist and set them
        if ( $services ) {
            $svc_array = array_map( 'trim', explode( ',', $services ) );
            $svc_array = array_filter( $svc_array );
            if ( ! empty( $svc_array ) ) {
                // Make sure taxonomy exists; register_taxonomy should already have been called by core plugin
                $term_ids = array();
                foreach ( $svc_array as $svc ) {
                    $svc_sanitized = sanitize_text_field( $svc );
                    $t = term_exists( $svc_sanitized, 'service_category' );
                    if ( $t === 0 || $t === null ) {
                        $new = wp_insert_term( $svc_sanitized, 'service_category' );
                        if ( ! is_wp_error( $new ) && isset( $new['term_id'] ) ) {
                            $term_ids[] = (int) $new['term_id'];
                        }
                    } else {
                        $term_ids[] = is_array( $t ) ? (int) $t['term_id'] : (int) $t;
                    }
                }
                if ( ! empty( $term_ids ) ) {
                    wp_set_post_terms( $post_id, $term_ids, 'service_category', false );
                }
            }
        }

        // Return the created post object
        return get_post( $post_id );
    }
}

/**
 * Render the contractor welcome card + optional creation form.
 * Shortcode: [contractor_welcome]
 */
if ( ! function_exists( 'rccc_contractor_welcome_shortcode' ) ) {
    function rccc_contractor_welcome_shortcode( $atts = array() ) {
        // require login for the welcome page
        if ( ! is_user_logged_in() ) {
            $login_url = wp_login_url( get_permalink() );
            return '<div class="rccc-welcome-notice" style="max-width:900px;margin:30px auto;padding:20px;border-radius:8px;background:#fff8db;border:1px solid #f1e1b8;text-align:center;">'
                 . '<p style="margin:0 0 8px;font-weight:600;">Please log in to view your contractor welcome page.</p>'
                 . '<p style="margin:0;"><a href="' . esc_url( $login_url ) . '" style="background:#047857;color:#fff;padding:10px 18px;border-radius:999px;text-decoration:none;">Log in</a></p>'
                 . '</div>';
        }

        $current_user = wp_get_current_user();
        $user_id = $current_user->ID;

        // Handle form submission (if present)
        $submission_result = null;
        if ( isset( $_POST['rccc_create_contractor_nonce'] ) ) {
            $submission_result = rccc_handle_welcome_form_submission();
            // If successful (WP_Post returned), redirect to avoid resubmission
            if ( $submission_result instanceof WP_Post ) {
                wp_safe_redirect( remove_query_arg( array( 'rccc_create_success' ) ) );
                exit;
            }
        }

        $contractor = rccc_get_contractor_for_user( $user_id );

        ob_start();

        // Basic container
        echo '<div class="rccc-welcome-shell" style="padding:40px 16px;">';
        echo '<div class="rccc-welcome-card" style="max-width:1000px;margin:0 auto;padding:32px;border-radius:12px;border:1px solid #e5e7eb;background:#fff;">';

        // Header
        echo '<h1 style="text-align:center;margin:0 0 18px;color:#14532d;font-size:28px;">Welcome to RainCity Contractor Connect</h1>';
        echo '<p style="text-align:center;color:#374151;margin:0 0 20px;">This is your contractor welcome page. Below you can view or create your public profile (used to match you with homeowners).</p>';

        if ( is_wp_error( $submission_result ) ) {
            echo '<div style="background:#fee7e7;border:1px solid #f5c2c2;padding:12px;border-radius:8px;margin:12px 0;color:#7a271a;">';
            echo esc_html( $submission_result->get_error_message() );
            echo '</div>';
        }

        if ( $contractor instanceof WP_Post ) {
            // Show contractor info
            $logo = rccc_get_contractor_logo_url( $contractor->ID );
            $title = get_the_title( $contractor->ID );
            $city = get_post_meta( $contractor->ID, 'rccc_city', true );
            $zips = get_post_meta( $contractor->ID, 'rccc_zips_csv', true );
            $email = get_post_meta( $contractor->ID, 'rccc_contact_email', true );
            $services = wp_get_post_terms( $contractor->ID, 'service_category', array( 'fields' => 'names' ) );

            echo '<div style="display:flex;gap:16px;align-items:center;flex-wrap:wrap;">';
            if ( $logo ) {
                echo '<div style="width:84px;height:84px;border-radius:12px;overflow:hidden;border:1px solid #e5e7eb;background:#fff;flex-shrink:0;">';
                echo '<img src="' . esc_url( $logo ) . '" alt="' . esc_attr( $title ) . ' logo" style="width:100%;height:100%;object-fit:contain;">';
                echo '</div>';
            } else {
                $initial = $title ? mb_substr( $title, 0, 1 ) : strtoupper( mb_substr( $current_user->display_name ?: $current_user->user_login, 0, 1 ) );
                echo '<div style="width:84px;height:84px;border-radius:999px;background:#047857;color:#fff;display:flex;align-items:center;justify-content:center;font-weight:700;font-size:30px;">' . esc_html( mb_strtoupper( $initial ) ) . '</div>';
            }

            echo '<div style="flex:1;min-width:220px;">';
            echo '<div style="font-size:18px;font-weight:700;color:#111827;">' . esc_html( $title ) . '</div>';
            if ( ! empty( $services ) ) {
                echo '<div style="margin-top:8px;"><strong>Services:</strong> ' . esc_html( implode( ', ', $services ) ) . '</div>';
            } else {
                echo '<div style="margin-top:8px;color:#6b7280;"><em>No services listed</em></div>';
            }
            $area = array();
            if ( $city ) $area[] = esc_html( $city );
            if ( $zips ) $area[] = esc_html( implode( ', ', array_map( 'trim', explode( ',', $zips ) ) ) );
            if ( ! empty( $area ) ) {
                echo '<div style="margin-top:8px;"><strong>Service area:</strong> ' . implode( ' · ', $area ) . '</div>';
            } else {
                echo '<div style="margin-top:8px;color:#6b7280;"><em>No service area listed</em></div>';
            }
            if ( $email ) {
                echo '<div style="margin-top:8px;">Contact email: <a href="mailto:' . esc_attr( $email ) . '">' . esc_html( $email ) . '</a></div>';
            }
            echo '</div>'; // flex:1
            echo '</div>'; // row

            // Links: View profile and Edit profile
            $profile_link = get_permalink( $contractor->ID );
            $edit_link = home_url( '/submit-contractor/' ); // existing ACF form
            echo '<div style="margin-top:20px;display:flex;gap:10px;flex-wrap:wrap;">';
            if ( $profile_link ) {
                echo '<a href="' . esc_url( $profile_link ) . '" style="background:#047857;color:#fff;padding:10px 18px;border-radius:999px;text-decoration:none;">View public profile</a>';
            }
            echo '<a href="' . esc_url( $edit_link ) . '" style="background:#0f172a;color:#fff;padding:10px 18px;border-radius:999px;text-decoration:none;">Edit your profile</a>';
            echo '</div>';
        } else {
            // No contractor found: show create form
            echo '<div style="margin-top:18px;">';
            echo '<h3 style="margin:0 0 12px;color:#0f172a;">Create your contractor profile</h3>';
            echo '<p style="margin:0 0 12px;color:#374151;">Fill out the short form below to create your public contractor profile. You can edit it later.</p>';

            // Pre-fill with user data
            $prefill_company = '';
            $prefill_email = $current_user->user_email;
            if ( ! empty( $current_user->display_name ) ) {
                $prefill_company = $current_user->display_name;
            } elseif ( ! empty( $current_user->user_login ) ) {
                $prefill_company = $current_user->user_login;
            }

            echo '<form method="post" style="max-width:720px;margin-top:12px;">';
            wp_nonce_field( 'rccc_create_contractor', 'rccc_create_contractor_nonce' );

            echo '<p><label style="display:block;font-weight:600;margin-bottom:6px;">Company name</label>';
            echo '<input type="text" name="rccc_company_name" value="' . esc_attr( $prefill_company ) . '" required style="width:100%;padding:10px;border:1px solid #e5e7eb;border-radius:8px;"></p>';

            echo '<p><label style="display:block;font-weight:600;margin-bottom:6px;">Contact email</label>';
            echo '<input type="email" name="rccc_contact_email" value="' . esc_attr( $prefill_email ) . '" required style="width:100%;padding:10px;border:1px solid #e5e7eb;border-radius:8px;"></p>';

            echo '<p><label style="display:block;font-weight:600;margin-bottom:6px;">Services (comma separated)</label>';
            echo '<input type="text" name="rccc_services" placeholder="e.g., Roofing, Siding, Gutters" style="width:100%;padding:10px;border:1px solid #e5e7eb;border-radius:8px;"></p>';

            echo '<p style="display:flex;gap:10px;"><span style="flex:1;"><label style="display:block;font-weight:600;margin-bottom:6px;">City</label>';
            echo '<input type="text" name="rccc_city" placeholder="City" style="width:100%;padding:10px;border:1px solid #e5e7eb;border-radius:8px;"></span>';
            echo '<span style="width:220px;"><label style="display:block;font-weight:600;margin-bottom:6px;">ZIPs (CSV)</label>';
            echo '<input type="text" name="rccc_zips" placeholder="98101,98102" style="width:100%;padding:10px;border:1px solid #e5e7eb;border-radius:8px;"></span></p>';

            echo '<p style="margin-top:8px;"><button type="submit" style="background:#047857;color:#fff;padding:10px 18px;border-radius:999px;border:none;cursor:pointer;">Create profile</button></p>';
            echo '</form>';

            echo '<p style="margin-top:12px;color:#6b7280;">Or if you already submitted via the contractor form, be sure you are logged in with the same email used for that submission.</p>';
            echo '</div>';
        }

        echo '</div>'; // card
        echo '</div>'; // shell

        return ob_get_clean();
    }

    add_shortcode( 'contractor_welcome', 'rccc_contractor_welcome_shortcode' );
}

/**
 * Also register a server-side block so Gutenberg users can insert the welcome card block.
 */
if ( function_exists( 'register_block_type' ) && ! function_exists( 'rccc_register_welcome_block' ) ) {
    function rccc_register_welcome_block() {
        register_block_type( 'rccc/contractor-welcome', array(
            'render_callback' => 'rccc_contractor_welcome_shortcode',
            'attributes'      => array(),
        ) );
    }
    add_action( 'init', 'rccc_register_welcome_block' );
}