<?php
/**
 * Plugin Name: RCCC Contractors – Profiles & Reviews
 * Description: Contractor CPT + taxonomy + ACF fields + front-end submit + star-rated reviews.
 * Version: 1.0.0
 */

if (!defined('ABSPATH')) exit;

class RCCC_Contractors_Profiles {

  public function __construct() {
    add_action('init',               [$this, 'register_types']);
    add_action('acf/init',           [$this, 'register_acf_fields']);   // ACF Pro
    add_shortcode('contractor_submit', [$this, 'shortcode_submit']);    // front-end form
    // Rating on comments (homeowner reviews)
    add_action('comment_form_after_fields', [$this, 'rating_field']);
    add_action('comment_form_logged_in_after', [$this, 'rating_field']);
    add_action('comment_post',       [$this, 'save_rating_meta']);
    add_action('transition_comment_status', [$this, 'recalc_post_rating'], 10, 3);
    add_action('wp_set_comment_status',     [$this, 'recalc_post_rating_status'], 10, 2);
  }

  /* ---------- Types ---------- */

  public function register_types() {
    register_taxonomy('service_category', 'contractor', [
      'labels' => [
        'name'          => 'Service Categories',
        'singular_name' => 'Service Category',
      ],
      'public'       => true,
      'show_in_rest' => true,
      'hierarchical' => true,
      'rewrite'      => ['slug' => 'services'],
    ]);

    register_post_type('contractor', [
      'label'         => 'Contractors',
      'public'        => true,
      'show_in_rest'  => true,
      'menu_icon'     => 'dashicons-hammer',
      'supports'      => ['title', 'editor', 'thumbnail', 'comments'],
      'has_archive'   => true,
      'rewrite'       => ['slug' => 'contractors'],
    ]);
  }

  /* ---------- ACF Field Group ---------- */
  public function register_acf_fields() {
    if (!function_exists('acf_add_local_field_group')) return;

    acf_add_local_field_group([
      'key'    => 'group_rccc_contractor',
      'title'  => 'Contractor Profile',
      'fields' => [
        [
          'key'   => 'field_rccc_logo',
          'label' => 'Company Logo',
          'name'  => 'rccc_logo',
          'type'  => 'image',
          'return_format' => 'id',
          'preview_size'  => 'medium',
          'instructions'  => 'Upload your company logo.',
        ],
        [
          'key'   => 'field_rccc_license',
          'label' => 'License Number',
          'name'  => 'rccc_license_no',
          'type'  => 'text',
          'required' => 1,
        ],
        [
          'key'   => 'field_rccc_bond',
          'label' => 'Bond',
          'name'  => 'rccc_bond',
          'type'  => 'text',
          'instructions' => 'Bond number or short description.',
        ],
        [
          'key'   => 'field_rccc_insurance',
          'label' => 'Insurance',
          'name'  => 'rccc_insurance',
          'type'  => 'text',
          'instructions' => 'Carrier / policy / notes.',
        ],
        [
          'key'   => 'field_rccc_tax',
          'label' => 'Service Categories',
          'name'  => 'rccc_services',
          'type'  => 'taxonomy',
          'taxonomy' => 'service_category',
          'field_type' => 'checkbox',
          'add_term'   => 1,
          'save_terms' => 1,
          'load_terms' => 1,
          'return_format' => 'object',
          'instructions' => 'Select all that apply.',
        ],
        [
          'key'   => 'field_rccc_portfolio',
          'label' => 'Portfolio (up to 2 projects)',
          'name'  => 'rccc_portfolio',
          'type'  => 'repeater',
          'min'   => 1,
          'max'   => 2,
          'layout'=> 'row',
          'sub_fields' => [
            [
              'key'   => 'field_rccc_proj_title',
              'label' => 'Project Title',
              'name'  => 'project_title',
              'type'  => 'text',
            ],
            [
              'key'   => 'field_rccc_before',
              'label' => 'Before (up to 3 images)',
              'name'  => 'before_gallery',
              'type'  => 'gallery',
              'min'   => 1,
              'max'   => 3,
              'return_format' => 'id',
              'preview_size'  => 'medium',
            ],
            [
              'key'   => 'field_rccc_after',
              'label' => 'After (up to 3 images)',
              'name'  => 'after_gallery',
              'type'  => 'gallery',
              'min'   => 1,
              'max'   => 3,
              'return_format' => 'id',
              'preview_size'  => 'medium',
            ],
          ],
        ],
      ],
      'location' => [[['param' => 'post_type', 'operator' => '==', 'value' => 'contractor']]],
      'position' => 'normal',
      'style'    => 'default',
    ]);
  }

  /* ---------- Front-end submit/edit ---------- */
  public function shortcode_submit($atts) {
    if (!function_exists('acf_form')) {
      return '<p>You need the ACF plugin active to use this form.</p>';
    }
    if (!is_user_logged_in()) {
      return '<p>You must be logged in to submit a contractor profile.</p>';
    }

    $atts = shortcode_atts([
      'post_id' => 0,   // pass ?contractor_id=123 to edit
    ], $atts);

    $edit_id = isset($_GET['contractor_id']) ? absint($_GET['contractor_id']) : absint($atts['post_id']);

    ob_start();
    echo '<div class="rccc-submit">';
    echo '<h2>' . ($edit_id ? 'Edit Contractor' : 'Add Contractor') . '</h2>';
    acf_form([
      'post_id'      => $edit_id ?: 'new_post',
      'new_post'     => [
        'post_type'   => 'contractor',
        'post_status' => 'publish',
      ],
      'uploader'     => 'wp',
      'submit_value' => $edit_id ? 'Update Contractor' : 'Create Contractor',
      'fields'       => [
        'field_rccc_logo',
        'field_rccc_license',
        'field_rccc_bond',
        'field_rccc_insurance',
        'field_rccc_tax',
        'field_rccc_portfolio',
      ],
    ]);
    echo '</div>';
    return ob_get_clean();
  }

  /* ---------- Reviews: add star rating to native comments ---------- */

  public function rating_field() {
    if (get_post_type() !== 'contractor') return;
    ?>
    <p class="comment-form-rating">
      <label for="rating">Rating <span class="required">*</span></label>
      <select name="rating" id="rating" required>
        <option value="">Select…</option>
        <option value="5">★★★★★ – Excellent</option>
        <option value="4">★★★★☆ – Very good</option>
        <option value="3">★★★☆☆ – Good</option>
        <option value="2">★★☆☆☆ – Fair</option>
        <option value="1">★☆☆☆☆ – Poor</option>
      </select>
    </p>
    <?php
  }

  public function save_rating_meta($comment_id) {
    if (empty($_POST['rating']) || get_post_type((int) $_POST['comment_post_ID']) !== 'contractor') return;
    $rating = max(1, min(5, (int) $_POST['rating']));
    add_comment_meta($comment_id, 'rating', $rating, true);
    $comment = get_comment($comment_id);
    if ($comment && $comment->comment_approved) {
      $this->update_contractor_rating((int) $comment->comment_post_ID);
    }
  }

  public function recalc_post_rating($new_status, $old_status, $comment) {
    if ($comment->comment_post_ID && get_post_type($comment->comment_post_ID) === 'contractor') {
      $this->update_contractor_rating((int) $comment->comment_post_ID);
    }
  }
  public function recalc_post_rating_status($comment_id, $new_status) {
    $c = get_comment($comment_id);
    if ($c && get_post_type($c->comment_post_ID) === 'contractor') {
      $this->update_contractor_rating((int) $c->comment_post_ID);
    }
  }

  private function update_contractor_rating($post_id) {
    $args = [
      'post_id' => $post_id,
      'status'  => 'approve',
      'meta_key'=> 'rating',
      'type'    => 'comment',
      'count'   => false,
    ];
    $comments = get_comments($args);
    $cnt = 0; $sum = 0;
    foreach ($comments as $c) {
      $r = (int) get_comment_meta($c->comment_ID, 'rating', true);
      if ($r) { $cnt++; $sum += $r; }
    }
    $avg = $cnt ? round($sum / $cnt, 1) : 0;
    update_post_meta($post_id, 'rccc_rating', $avg);
    update_post_meta($post_id, 'rccc_reviews', $cnt);
  }
}

new RCCC_Contractors_Profiles();
