<?php
/**
 * Plugin Name: RCCC Contractor Capabilities
 * Description: Registers custom capabilities for the contractor CPT and creates a lightweight "contractor" role that can only manage contractor posts (not blog posts).
 * Version: 1.0.0
 * Author: RCCC
 *
 * Install:
 * 1. Save this file to wp-content/plugins/rccc-contractor-capabilities.php
 * 2. Activate the plugin in WP Admin -> Plugins.
 *
 * Notes:
 * - This plugin re-registers the 'contractor' post type on init with custom capabilities (map_meta_cap => true).
 *   If you already register 'contractor' elsewhere, this call will override that registration to ensure capabilities are applied.
 * - On activation it creates the 'contractor' role and grants it only the capabilities needed to manage contractor CPT items.
 * - Admins are granted all contractor capabilities so they can manage contractor posts.
 * - This file does not give contractors permissions to create/edit standard blog posts.
 */

/* Prevent direct access */
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/* Key names */
define( 'RCCC_CONTRACTOR_ROLE_KEY', 'contractor' );

/* Capabilities mapping for the contractor CPT */
function rccc_contractor_caps_map() {
    return array(
        'edit_post'             => 'edit_contractor',
        'read_post'             => 'read_contractor',
        'delete_post'           => 'delete_contractor',
        'edit_posts'            => 'edit_contractors',
        'edit_others_posts'     => 'edit_others_contractors',
        'publish_posts'         => 'publish_contractors',
        'read_private_posts'    => 'read_private_contractors',
        'delete_posts'          => 'delete_contractors',
        'delete_private_posts'  => 'delete_private_contractors',
        'delete_published_posts'=> 'delete_published_contractors',
        'delete_others_posts'   => 'delete_others_contractors',
        'edit_private_posts'    => 'edit_private_contractors',
        'edit_published_posts'  => 'edit_published_contractors',
    );
}

/* Re-register contractor CPT with capability mapping so WP enforces meta caps */
add_action( 'init', 'rccc_register_contractor_cpt_with_caps', 20 );
function rccc_register_contractor_cpt_with_caps() {
    $caps = rccc_contractor_caps_map();

    $args = array(
        'label'              => 'Contractors',
        'public'             => true,
        'show_in_rest'       => true,
        'menu_icon'          => 'dashicons-hammer',
        'supports'           => array( 'title', 'editor', 'thumbnail', 'custom-fields', 'author', 'comments' ),
        'has_archive'        => true,
        'rewrite'            => array( 'slug' => 'contractors' ),
        'capability_type'    => 'contractor',
        'capabilities'       => $caps,
        'map_meta_cap'       => true,
    );

    // Register (this will override earlier registration if present)
    register_post_type( 'contractor', $args );
}

/* Create the contractor role and give it only the contractor-related caps */
function rccc_add_contractor_role_and_caps() {
    $role_key = RCCC_CONTRACTOR_ROLE_KEY;

    $contractor_caps = array(
        // Minimal base capabilities
        'read' => true,
        'upload_files' => true,
        // Contractor CPT specific capabilities (allow managing their own contractor post)
        'edit_contractor' => true,
        'read_contractor' => true,
        'delete_contractor' => true,

        // Allow managing lists and publishing their own contractor item
        'edit_contractors' => true,
        'publish_contractors' => true,
        'edit_published_contractors' => true,
        'edit_private_contractors' => true,
        'read_private_contractors' => true,
        'delete_published_contractors' => true,
    );

    // Add role if missing
    if ( ! get_role( $role_key ) ) {
        add_role( $role_key, 'Contractor', $contractor_caps );
    } else {
        // Ensure the caps are set on existing role
        $role = get_role( $role_key );
        if ( $role ) {
            foreach ( $contractor_caps as $cap => $grant ) {
                if ( $grant ) {
                    $role->add_cap( $cap );
                } else {
                    $role->remove_cap( $cap );
                }
            }
        }
    }

    // Give all contractor-related caps to administrators so they can manage contractor posts
    $admin = get_role( 'administrator' );
    if ( $admin ) {
        $caps_map = rccc_contractor_caps_map();
        // base read/upload
        $admin->add_cap( 'read' );
        $admin->add_cap( 'upload_files' );
        // add all mapped caps
        foreach ( $caps_map as $c ) {
            $admin->add_cap( $c );
        }
        // add convenience caps used above
        $admin->add_cap( 'edit_contractors' );
        $admin->add_cap( 'publish_contractors' );
        $admin->add_cap( 'edit_published_contractors' );
        $admin->add_cap( 'edit_private_contractors' );
        $admin->add_cap( 'read_private_contractors' );
        $admin->add_cap( 'delete_published_contractors' );
    }
}

/* Activation hook: add role and caps */
register_activation_hook( __FILE__, 'rccc_on_activation_add_role_caps' );
function rccc_on_activation_add_role_caps() {
    // Ensure CPT is registered (our init will run during normal request; on activation we call directly)
    rccc_register_contractor_cpt_with_caps();
    rccc_add_contractor_role_and_caps();
}

/* Deactivation: we do not remove the role or caps to avoid destructive changes.
   If you want to clean up, implement it carefully and run manually. */

/* Optional admin notice to confirm plugin is active and role exists */
add_action( 'admin_notices', 'rccc_contract_caps_admin_notice' );
function rccc_contract_caps_admin_notice() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    // Only show once / only when plugin active
    if ( get_transient( 'rccc_contract_caps_shown' ) ) {
        return;
    }
    if ( get_role( RCCC_CONTRACTOR_ROLE_KEY ) ) {
        printf(
            '<div class="updated"><p>%s</p></div>',
            esc_html__( 'RCCC Contractor Capabilities active — "contractor" role created and contractor CPT mapped to custom capabilities.', 'rccc' )
        );
        set_transient( 'rccc_contract_caps_shown', 1, 30 );
    } else {
        printf(
            '<div class="notice notice-warning"><p>%s</p></div>',
            esc_html__( 'RCCC Contractor Capabilities active — but the "contractor" role was not created. Check permissions.', 'rccc' )
        );
        set_transient( 'rccc_contract_caps_shown', 1, 30 );
    }
}