<?php
/**
 * Plugin Name: RainCity Contractor Connect – Core (Settings)
 * Description: Contractors directory + homeowner lead intake + matching & email routing + reviews + single page layout with right-rail sidebar.
 * Version: 0.6.3
 * Author: RCCC
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/* Safety for registration redirect path constant */
if ( ! defined( 'RCCC_REG_REDIRECT_PATH' ) ) {
    define( 'RCCC_REG_REDIRECT_PATH', '/6718/submit-contractor/' );
}

class RCCC_Core_Settings {
    const VERSION           = '0.6.3';
    const DEFAULT_FORM_ID   = 1;
    const OPTION_FORM_ID    = 'rccc_form_id';
    const OPTION_N_MATCH    = 'rccc_n_match';
    const OPTION_ACCEPT_MIN = 'rccc_accept_min';

    public function __construct() {
        // Data model / CPTs
        add_action( 'init', [ $this, 'register_types' ] );

        // Lifecycle
        register_activation_hook( __FILE__,  [ $this, 'activate' ] );
        register_deactivation_hook( __FILE__,[ $this, 'deactivate' ] );

        // Settings UI
        add_action( 'admin_menu',  [ $this, 'admin_menu' ] );
        add_action( 'admin_init',  [ $this, 'register_settings' ] );

        // Directory + token actions
        add_shortcode( 'rccc_contractor_directory', [ $this, 'shortcode_contractor_directory' ] );
        add_action( 'template_redirect',            [ $this, 'handle_token_actions' ] );

        // Template routing (classic themes only)
        add_filter( 'template_include', [ $this, 'route_contractor_templates' ], 99 );

        // Lead intake (homeowner WPForms)
        add_action( 'wpforms_process_complete', [ $this, 'on_wpforms_complete' ], 10, 4 );
    }

    /* ---------- Activation / Types ---------- */

    public function activate() {
        $this->register_types();
        if ( get_option( self::OPTION_FORM_ID )    === false ) update_option( self::OPTION_FORM_ID,    self::DEFAULT_FORM_ID );
        if ( get_option( self::OPTION_N_MATCH )    === false ) update_option( self::OPTION_N_MATCH,    3 );
        if ( get_option( self::OPTION_ACCEPT_MIN ) === false ) update_option( self::OPTION_ACCEPT_MIN, 30 );
        flush_rewrite_rules();
    }

    public function deactivate() {
        flush_rewrite_rules();
    }

    public function register_types() {
        register_post_type( 'contractor', [
            'label'        => 'Contractors',
            'public'       => true,
            'show_in_rest' => true,
            'menu_icon'    => 'dashicons-hammer',
            'supports'     => [ 'title', 'editor', 'thumbnail', 'custom-fields', 'author', 'comments' ],
            'has_archive'  => true,
            'rewrite'      => [ 'slug' => 'contractors' ],
        ] );

        register_taxonomy( 'service_category', 'contractor', [
            'label'        => 'Service Categories',
            'public'       => true,
            'show_in_rest' => true,
            'hierarchical' => true,
            'rewrite'      => [ 'slug' => 'services' ],
        ] );

        register_post_type( 'project', [
            'label'    => 'Projects (Leads)',
            'public'   => false,
            'show_ui'  => true,
            'supports' => [ 'title', 'editor', 'custom-fields' ],
        ] );

        // Contractor meta (restrict REST edits to users who can edit the post)
        foreach ( [ 'rccc_zip','rccc_city','rccc_contact_email','rccc_phone','rccc_zips_csv','rccc_rating','rccc_reviews','rccc_license_no','rccc_insurance_expires' ] as $k ) {
            register_post_meta( 'contractor', $k, [
                'show_in_rest'  => true,
                'single'        => true,
                'type'          => 'string',
                'auth_callback' => function ( $allowed, $meta_key, $post_id ) {
                    // Only allow editing via REST to users who can edit the post
                    return current_user_can( 'edit_post', $post_id );
                },
            ] );
        }

        // Lead meta (project) - administrative only (kept as before, hidden from REST by default)
        foreach ( [ 'name','email','phone','zip','category','budget','timing','desc','matched_ids','accepted_ids','tokens' ] as $k ) {
            register_post_meta( 'project', 'rccc_' . $k, [
                'show_in_rest'  => false,
                'single'        => true,
                'type'          => 'string',
                'auth_callback' => '__return_true',
            ] );
        }
    }

    /* ---------- Settings UI ---------- */

    public function admin_menu() {
        add_options_page( 'RCCC Settings', 'RCCC', 'manage_options', 'rccc-settings', [ $this, 'settings_page' ] );
    }

    public function register_settings() {
        register_setting( 'rccc', self::OPTION_FORM_ID,    [ 'type' => 'integer', 'sanitize_callback' => 'absint' ] );
        register_setting( 'rccc', self::OPTION_N_MATCH,    [ 'type' => 'integer', 'sanitize_callback' => 'absint' ] );
        register_setting( 'rccc', self::OPTION_ACCEPT_MIN, [ 'type' => 'integer', 'sanitize_callback' => 'absint' ] );

        add_settings_section( 'rccc_main', 'Lead Settings', function () {
            echo '<p>Configure your WPForms form ID and lead routing.</p>';
        }, 'rccc' );

        add_settings_field( self::OPTION_FORM_ID, 'WPForms Form ID', function () {
            $v = get_option( self::OPTION_FORM_ID, self::DEFAULT_FORM_ID );
            echo '<input type="number" name="' . esc_attr( self::OPTION_FORM_ID ) . '" value="' . esc_attr( $v ) . '" min="1" class="small-text" />';
        }, 'rccc', 'rccc_main' );

        add_settings_field( self::OPTION_N_MATCH, 'Contractors per lead', function () {
            $v = get_option( self::OPTION_N_MATCH, 3 );
            echo '<input type="number" name="' . esc_attr( self::OPTION_N_MATCH ) . '" value="' . esc_attr( $v ) . '" min="1" max="10" class="small-text" />';
        }, 'rccc', 'rccc_main' );

        add_settings_field( self::OPTION_ACCEPT_MIN, 'Accept window (minutes)', function () {
            $v = get_option( self::OPTION_ACCEPT_MIN, 30 );
            echo '<input type="number" name="' . esc_attr( self::OPTION_ACCEPT_MIN ) . '" value="' . esc_attr( $v ) . '" min="5" max="240" class="small-text" />';
        }, 'rccc', 'rccc_main' );
    }

    public function settings_page() { ?>
        <div class="wrap">
            <h1>RainCity Contractor Connect – Settings</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'rccc' );
                do_settings_sections( 'rccc' );
                submit_button();
                ?>
            </form>
        </div>
    <?php }

    /* ---------- Helpers ---------- */

    private function form_id() {
        $id = absint( get_option( self::OPTION_FORM_ID, self::DEFAULT_FORM_ID ) );
        return $id > 0 ? $id : self::DEFAULT_FORM_ID;
    }

    private function n_match() {
        $n = absint( get_option( self::OPTION_N_MATCH, 3 ) );
        return $n > 0 ? $n : 3;
    }

    private function accept_window_min() {
        $m = absint( get_option( self::OPTION_ACCEPT_MIN, 30 ) );
        return $m > 0 ? $m : 30;
    }

    /**
     * PHP < 7.3-compatible replacement for array_key_last()
     */
    private function array_key_last_compat( array $arr ) {
        if ( function_exists( 'array_key_last' ) ) {
            return array_key_last( $arr );
        }
        if ( empty( $arr ) ) {
            return null;
        }
        end( $arr );
        return key( $arr );
    }

    private function extract_zip_from_string( $val ) {
        if ( is_array( $val ) ) {
            $val = implode( ' ', array_map( 'strval', $val ) );
        }
        $val = (string) $val;
        if ( preg_match_all( '/\b(\d{5})(?:-\d{4})?\b/', $val, $m ) && ! empty( $m[1] ) ) {
            $k = $this->array_key_last_compat( $m[1] );
            if ( $k === null ) return '';
            return $m[1][ $k ];
        }
        return '';
    }

    private function extract_zip_from_fields( array $fields ) {
        $primary   = [ 'zip', 'postal code', 'postal', 'postcode' ];
        $secondary = [ 'address','address line','address line 1','address line 2','city state zip','csz' ];

        foreach ( $fields as $f ) {
            if ( ! isset( $f['name'] ) ) continue;
            $name = strtolower( trim( (string) $f['name'] ) );
            if ( in_array( $name, $primary, true ) ) {
                if ( isset( $f['value'] ) && is_array( $f['value'] ) ) {
                    foreach ( [ 'postal','zip','postcode' ] as $k ) {
                        if ( ! empty( $f['value'][ $k ] ) ) {
                            return preg_replace( '/\D/', '', (string) $f['value'][ $k ] );
                        }
                    }
                }
                if ( isset( $f['value'] ) && is_string( $f['value'] ) ) {
                    $z = $this->extract_zip_from_string( $f['value'] );
                    if ( $z ) return $z;
                }
            }
        }

        foreach ( $fields as $f ) {
            if ( ! empty( $f['type'] ) && $f['type'] === 'address' && ! empty( $f['value'] ) && is_array( $f['value'] ) ) {
                foreach ( [ 'postal','zip','postcode' ] as $k ) {
                    if ( ! empty( $f['value'][ $k ] ) ) {
                        return preg_replace( '/\D/', '', (string) $f['value'][ $k ] );
                    }
                }
            }
        }

        foreach ( $fields as $f ) {
            if ( ! isset( $f['name'] ) ) continue;
            $name = strtolower( trim( (string) $f['name'] ) );
            if ( in_array( $name, $secondary, true ) && isset( $f['value'] ) ) {
                $z = $this->extract_zip_from_string( $f['value'] );
                if ( $z ) return $z;
            }
        }
        return '';
    }

    private function find_contractors( $category_name, $zip, $limit = 9 ) {
        $args = [
            'post_type'      => 'contractor',
            'posts_per_page' => $limit,
            'orderby'        => 'meta_value_num',
            'meta_key'       => 'rccc_rating',
            'order'          => 'DESC',
            'post_status'    => 'publish',
        ];

        if ( ! empty( $category_name ) ) {
            $cat_name = trim( (string) $category_name );
            $term     = get_term_by( 'name', $cat_name, 'service_category' );
            if ( ! $term || is_wp_error( $term ) ) {
                $term = get_term_by( 'slug', sanitize_title( $cat_name ), 'service_category' );
            }
            if ( $term && ! is_wp_error( $term ) ) {
                $args['tax_query'] = [[
                    'taxonomy' => 'service_category',
                    'field'    => 'term_id',
                    'terms'    => [ $term->term_id ],
                ]];
            }
        }

        if ( ! empty( $zip ) ) {
            $zip = preg_replace( '/[^0-9]/', '', $zip );
            $args['meta_query'][] = [
                'key'     => 'rccc_zips_csv',
                'value'   => $zip,
                'compare' => 'LIKE',
            ];
        }

        return get_posts( $args );
    }

    /* ---------- Directory Shortcode ---------- */

    public function shortcode_contractor_directory( $atts ) {
        $atts     = shortcode_atts( [ 'per_page' => 12 ], $atts );
        $category = isset( $_GET['category'] ) ? sanitize_text_field( $_GET['category'] ) : '';
        $zip      = isset( $_GET['zip'] )      ? preg_replace( '/[^0-9]/', '', $_GET['zip'] ) : '';

        $tax_query  = [];
        $meta_query = [];

        if ( $category ) $tax_query[]  = [ 'taxonomy' => 'service_category', 'field' => 'name', 'terms' => $category ];
        if ( $zip )      $meta_query[] = [ 'key' => 'rccc_zips_csv', 'value' => $zip, 'compare' => 'LIKE' ];

        $q = new WP_Query( [
            'post_type'      => 'contractor',
            'posts_per_page' => (int) $atts['per_page'],
            'tax_query'      => $tax_query ?: null,
            'meta_query'     => $meta_query ?: null,
        ] );

        ob_start(); ?>
        <form method="get" class="rccc-filters" style="display:flex;gap:8px;flex-wrap:wrap;margin-bottom:12px;">
            <input type="text" name="zip"      value="<?php echo esc_attr( $zip ); ?>"      placeholder="ZIP (e.g., 98101)">
            <input type="text" name="category" value="<?php echo esc_attr( $category ); ?>" placeholder="Category (e.g., Roofing)">
            <button type="submit">Filter</button>
        </form>

        <div class="rccc-grid" style="display:grid;grid-template-columns:repeat(auto-fill,minmax(260px,1fr));gap:14px;">
        <?php
        if ( $q->have_posts() ) :
            while ( $q->have_posts() ) : $q->the_post();
                $id        = get_the_ID();
                $rating    = get_post_meta( $id, 'rccc_rating', true );
                $reviews   = get_post_meta( $id, 'rccc_reviews', true );
                $city      = get_post_meta( $id, 'rccc_city', true );
                $zips      = get_post_meta( $id, 'rccc_zips_csv', true );
                $cats      = wp_get_post_terms( $id, 'service_category', [ 'fields' => 'names' ] );
                $permalink = get_the_permalink();

                $lead_page = get_page_by_path( 'get-matched' );
                $lead_url  = trailingslashit( $lead_page ? get_permalink( $lead_page ) : home_url( '/get-matched/' ) );
                $first_cat = $category ?: ( $cats[0] ?? '' );
                $first_zip = $zip ?: ( $zips ? trim( explode( ',', $zips )[0] ) : '' );
                ?>
                <div class="rccc-card" style="border:1px solid #e2e8f0;border-radius:14px;padding:12px;background:#fff;">
                    <div style="display:flex;justify-content:space-between;align-items:flex-start;gap:8px;">
                        <div>
                            <h3 style="margin:0 0 6px;font-size:18px;font-weight:600;">
                                <a href="<?php echo esc_url( $permalink ); ?>" style="text-decoration:none;color:inherit;"><?php the_title(); ?></a>
                            </h3>
                            <div style="font-size:12px;color:#475569;">
                                📍 <?php echo esc_html( $city ); ?>
                                <?php if ( $rating !== '' ) : ?>
                                    · ⭐ <?php echo esc_html( $rating ); ?> (<?php echo esc_html( $reviews ); ?>)
                                <?php endif; ?>
                            </div>
                            <div style="margin-top:6px;font-size:12px;color:#334155;">Services: <?php echo esc_html( implode( ', ', $cats ) ); ?></div>
                            <div style="margin-top:6px;font-size:12px;color:#64748b;">ZIPs: <?php echo esc_html( $zips ); ?></div>
                        </div>
                        <div>
                            <a href="<?php echo esc_url( add_query_arg( array_filter( [ 'category' => $first_cat ?: null, 'zip' => $first_zip ?: null ] ), $lead_url ) ); ?>"
                               style="display:inline-block;background:#047857;color:#fff;padding:8px 10px;border-radius:10px;text-decoration:none;font-size:13px;">Request quote</a>
                            <a href="<?php echo esc_url( $permalink ); ?>"
                               style="display:inline-block;background:#0f172a;color:#fff;padding:8px 10px;border-radius:10px;text-decoration:none;font-size:13px;margin-left:8px;">View profile</a>
                        </div>
                    </div>
                </div>
            <?php
            endwhile;
        else :
            echo '<div>No contractors found. Try a different ZIP or category.</div>';
        endif;
        wp_reset_postdata();
        ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /* ---------- Token actions (accept / decline) ---------- */

    private function issue_token( $project_id, $contractor_id ) {
        $tokens = get_post_meta( $project_id, 'rccc_tokens', true );
        $tokens = $tokens ? json_decode( $tokens, true ) : [];
        $token  = wp_generate_password( 20, false, false );
        $tokens[ $token ] = [
            'contractor_id' => $contractor_id,
            'issued'        => time(),
        ];
        update_post_meta( $project_id, 'rccc_tokens', wp_json_encode( $tokens ) );
        return $token;
    }

    private function validate_token( $project_id, $contractor_id, $token ) {
        $tokens = get_post_meta( $project_id, 'rccc_tokens', true );
        $tokens = $tokens ? json_decode( $tokens, true ) : [];
        if ( ! isset( $tokens[ $token ] ) ) return false;
        $row = $tokens[ $token ];
        if ( (int) $row['contractor_id'] !== (int) $contractor_id ) return false;
        if ( ( time() - (int) $row['issued'] ) > ( $this->accept_window_min() * 60 ) ) return false;
        return true;
    }

    public function handle_token_actions() {
        if ( ! isset( $_GET['rccc'] ) ) return;

        $action        = sanitize_key( $_GET['rccc'] );
        $project_id    = isset( $_GET['project'] )    ? (int) $_GET['project']    : 0;
        $contractor_id = isset( $_GET['contractor'] ) ? (int) $_GET['contractor'] : 0;
        $token         = isset( $_GET['token'] )      ? sanitize_text_field( $_GET['token'] ) : '';

        if ( ! $project_id || ! $contractor_id || ! $token ) return;
        if ( ! $this->validate_token( $project_id, $contractor_id, $token ) ) wp_die( 'Sorry, this link is invalid or expired.' );

        // Consume the token immediately so it cannot be reused
        $tokens = get_post_meta( $project_id, 'rccc_tokens', true );
        $tokens = $tokens ? json_decode( $tokens, true ) : [];
        if ( isset( $tokens[ $token ] ) ) {
            unset( $tokens[ $token ] );
            update_post_meta( $project_id, 'rccc_tokens', wp_json_encode( $tokens ) );
        }

        $accepted     = get_post_meta( $project_id, 'rccc_accepted_ids', true );
        $accepted_ids = $accepted ? array_filter( array_map( 'intval', explode( ',', $accepted ) ) ) : [];

        if ( $action === 'accept' ) {
            $accepted_ids[] = $contractor_id;
            update_post_meta( $project_id, 'rccc_accepted_ids', implode( ',', array_unique( $accepted_ids ) ) );

            // Notify homeowner of acceptance (if email present)
            $homeowner_email = get_post_meta( $project_id, 'rccc_email', true );
            if ( is_email( $homeowner_email ) ) {
                $contractor_name = get_the_title( $contractor_id );
                $contractor_email = get_post_meta( $contractor_id, 'rccc_contact_email', true );
                $contractor_phone = get_post_meta( $contractor_id, 'rccc_phone', true );

                $subject = 'A contractor has accepted your lead';
                $body    = "Hello,\n\n"
                         . "Good news — {$contractor_name} has accepted your lead.\n\n"
                         . "Contractor contact: " . ( $contractor_email ? $contractor_email : 'N/A' ) . "\n"
                         . "Phone: " . ( $contractor_phone ? $contractor_phone : 'N/A' ) . "\n\n"
                         . "They will reach out to you shortly regarding your project.\n\n"
                         . "Thanks,\nRainCity Contractor Connect";

                wp_mail( $homeowner_email, $subject, $body, [ 'Content-Type: text/plain; charset=UTF-8' ] );
            }

            wp_die( 'Thanks! You have accepted this lead. The homeowner has been notified.' );
        } elseif ( $action === 'decline' ) {
            wp_die( 'You have declined this lead. Thanks for the quick response.' );
        }
    }

    /* ---------- Template routing ---------- */

    public function route_contractor_templates( $template ) {
        if ( function_exists( 'wp_is_block_theme' ) && wp_is_block_theme() ) return $template;

        if ( is_singular( 'contractor' ) ) {
            $file = plugin_dir_path( __FILE__ ) . 'templates/single-contractor.php';
            if ( file_exists( $file ) ) return $file;
        }

        if ( is_post_type_archive( 'contractor' ) ) {
            $file = plugin_dir_path( __FILE__ ) . 'templates/archive-contractor.php';
            if ( file_exists( $file ) ) return $file;
        }

        return $template;
    }

    /* ---------- WPForms hook (homeowner lead form) ---------- */

    public function on_wpforms_complete( $fields, $entry, $form_data, $entry_id ) {
        if ( (int) $form_data['id'] !== $this->form_id() ) return;

        $get_any = function ( array $labels ) use ( $fields ) {
            foreach ( $fields as $f ) {
                if ( ! isset( $f['name'] ) ) continue;
                $name = strtolower( trim( (string) $f['name'] ) );
                foreach ( $labels as $label ) {
                    if ( $name === strtolower( trim( (string) $label ) ) ) {
                        $val = isset( $f['value'] ) ? $f['value'] : '';
                        return is_string( $val ) ? trim( $val ) : $val;
                    }
                }
            }
            return '';
        };

        $name   = (string) $get_any( [ 'Name','Full Name' ] );
        $email  = (string) $get_any( [ 'Email','Email Address' ] );
        $phone  = (string) $get_any( [ 'Phone','Phone Number','Telephone' ] );

        $zip = $this->extract_zip_from_fields( $fields );
        if ( ! $zip && isset( $_REQUEST['zip'] ) ) $zip = preg_replace( '/\D/', '', (string) $_REQUEST['zip'] );

        $category = (string) $get_any( [ 'Project Type','Service Needed','Category','Service' ] );
        if ( ! $category && isset( $_REQUEST['category'] ) ) $category = sanitize_text_field( (string) $_REQUEST['category'] );

        $desc   = (string) $get_any( [ 'Description','Project Description','Details','Message' ] );
        $budget = (string) $get_any( [ 'Budget' ] );
        $timing = (string) $get_any( [ 'Timing','Timeline' ] );

        // Create project post as private to avoid accidental exposure
        $project_id = wp_insert_post( [
            'post_type'   => 'project',
            'post_title'  => wp_strip_all_tags( ( $category ?: 'Lead' ) . ' · ' . ( $zip ?: 'No ZIP' ) . ' · ' . current_time( 'mysql' ) ),
            'post_status' => 'private',
            'post_content'=> $desc,
        ] );

        if ( $project_id && ! is_wp_error( $project_id ) ) {
            update_post_meta( $project_id, 'rccc_name',    $name );
            update_post_meta( $project_id, 'rccc_email',   $email );
            update_post_meta( $project_id, 'rccc_phone',   $phone );
            update_post_meta( $project_id, 'rccc_zip',     $zip );
            update_post_meta( $project_id, 'rccc_category',$category );
            update_post_meta( $project_id, 'rccc_budget',  $budget );
            update_post_meta( $project_id, 'rccc_timing',  $timing );
            update_post_meta( $project_id, 'rccc_desc',    $desc );
        }

        $matched = $this->find_contractors( $category, $zip, $this->n_match() * 3 );
        if ( empty( $matched ) && ! empty( $zip ) )      $matched = $this->find_contractors( '',        $zip, $this->n_match() * 3 );
        if ( empty( $matched ) && ! empty( $category ) ) $matched = $this->find_contractors( $category, '',   $this->n_match() * 3 );

        $matched_ids = wp_list_pluck( $matched, 'ID' );
        update_post_meta( $project_id, 'rccc_matched_ids', implode( ',', $matched_ids ) );

        $sent_to = [];
        foreach ( array_slice( (array) $matched, 0, $this->n_match() ) as $post ) {
            $to = get_post_meta( $post->ID, 'rccc_contact_email', true );
            if ( ! $to || ! is_email( $to ) ) continue;

            $token   = $this->issue_token( $project_id, $post->ID );
            $accept  = add_query_arg( [ 'rccc' => 'accept',  'project' => $project_id, 'contractor' => $post->ID, 'token' => $token ], home_url( '/' ) );
            $decline = add_query_arg( [ 'rccc' => 'decline', 'project' => $project_id, 'contractor' => $post->ID, 'token' => $token ], home_url( '/' ) );

            $subject = 'New local lead: ' . ( $category ?: 'Project' ) . ' in ' . ( $zip ?: 'your area' );
            $body    = "You have a new lead from RainCity Contractor Connect.\n\n"
                     . "Project: " . ( $category ?: 'N/A' ) . "\n"
                     . "ZIP: "     . ( $zip ?: 'N/A' ) . "\n"
                     . "Budget: "  . ( $budget ?: 'N/A' ) . "\n"
                     . "Timing: "  . ( $timing ?: 'N/A' ) . "\n"
                     . "Description: " . ( $desc ?: 'N/A' ) . "\n\n"
                     . "Homeowner: " . ( $name ?: 'N/A' ) . ", " . ( $email ?: 'N/A' ) . ", " . ( $phone ?: 'N/A' ) . "\n\n"
                     . "Accept: $accept\nDecline: $decline\n\n"
                     . "This link expires in " . $this->accept_window_min() . " minutes.";

            wp_mail( $to, $subject, $body, [ 'Content-Type: text/plain; charset=UTF-8' ] );
            $sent_to[] = $to;
        }

        if ( empty( $sent_to ) ) {
            $admin_email = get_option( 'admin_email' );
            wp_mail(
                $admin_email,
                'RCCC: No contractors matched this lead',
                'Category: ' . ( $category ?: 'N/A' ) . "\nZIP: " . ( $zip ?: 'N/A' ) . "\nProject ID: " . $project_id
            );
        }
    }
}

/* Boot the plugin */
new RCCC_Core_Settings();

/* === Right-rail sidebar for ads/publicity === */
add_action( 'widgets_init', function () {
    register_sidebar( [
        'name'          => 'RCCC Contractor Sidebar',
        'id'            => 'rccc_contractor_sidebar',
        'description'   => 'Shown on the right side of single contractor pages. Add Image, Custom HTML, Shortcode, etc.',
        'before_widget' => '<div class="rccc-ad">',
        'after_widget'  => '</div>',
        'before_title'  => '<h4 class="rccc-ad-title">',
        'after_title'   => '</h4>',
    ] );
} );

/* --- Remove theme comment UIs on contractor pages (prevents duplicates) --- */
add_filter( 'render_block', function ( $content, $block ) {
    if ( ! is_singular( 'contractor' ) || empty( $block['blockName'] ) ) return $content;
    $kill = [
        'core/comments','core/post-comments','core/comments-title','core/comments-query-loop',
        'core/comment-template','core/comments-pagination','core/post-comments-form','core/post-comments-count',
    ];
    return in_array( $block['blockName'], $kill, true ) ? '' : $content;
}, 9, 2 );

/* Keep comments open for contractor posts (for reviews) */
add_filter( 'comments_open', function ( $open, $post_id ) {
    return ( get_post_type( $post_id ) === 'contractor' ) ? true : $open;
}, 10, 2 );

/* ---------- SINGLE CONTRACTOR CONTENT (profile + reviews) ---------- */
function rccc_render_single_contractor_card( $content ) {
    if ( ! is_singular( 'contractor' ) || ! in_the_loop() || ! is_main_query() ) {
        return $content;
    }

    $id      = get_the_ID();
    $title   = get_the_title( $id );
    $city    = get_post_meta( $id, 'rccc_city', true );
    $zips    = get_post_meta( $id, 'rccc_zips_csv', true );
    $rating  = get_post_meta( $id, 'rccc_rating', true );
    $reviews = get_post_meta( $id, 'rccc_reviews', true );
    $license = get_post_meta( $id, 'rccc_license_no', true );
    $insExp  = get_post_meta( $id, 'rccc_insurance_expires', true );
    $email   = get_post_meta( $id, 'rccc_contact_email', true );
    $cats    = wp_get_post_terms( $id, 'service_category', [ 'fields' => 'names' ] );

    // Logo: prefer ACF rccc_contractor_logo, then fall back to featured image
    $logo_url = '';

    if ( function_exists( 'get_field' ) ) {
        $acf_logo = get_field( 'rccc_contractor_logo', $id );

        if ( is_array( $acf_logo ) && ! empty( $acf_logo['ID'] ) ) {
            $logo_url = wp_get_attachment_image_url( (int) $acf_logo['ID'], 'medium' );
        } elseif ( is_numeric( $acf_logo ) ) {
            $logo_url = wp_get_attachment_image_url( (int) $acf_logo, 'medium' );
        } elseif ( is_string( $acf_logo ) && filter_var( $acf_logo, FILTER_VALIDATE_URL ) ) {
            $logo_url = $acf_logo;
        }
    }

    if ( ! $logo_url ) {
        $fid = get_post_thumbnail_id( $id );
        if ( $fid ) {
            $logo_url = wp_get_attachment_image_url( $fid, 'medium' );
        }
    }

    $lead_page = get_page_by_path( 'get-matched' );
    $lead_url  = trailingslashit( $lead_page ? get_permalink( $lead_page ) : home_url( '/get-matched/' ) );
    $first_cat = $cats ? $cats[0] : '';
    $first_zip = $zips ? trim( explode( ',', $zips )[0] ) : '';

    $desc_raw = get_post_field( 'post_content', $id );
    $desc     = wpautop( wp_kses_post( $desc_raw ) );

    ob_start(); ?>
    <div class="rccc-single-card">
        <div class="rccc-header">
            <?php if ( $logo_url ) : ?>
                <img class="rccc-logo" src="<?php echo esc_url( $logo_url ); ?>" alt="<?php echo esc_attr( $title ); ?> logo">
            <?php else : ?>
                <div class="rccc-logo rccc-logo--placeholder" aria-hidden="true">R</div>
            <?php endif; ?>
            <h2 class="rccc-name"><?php echo esc_html( $title ); ?></h2>
        </div>

        <div class="rccc-top">
            <div class="rccc-badges">
                <?php if ( $city ) : ?>
                    <span class="rccc-pill">📍 <?php echo esc_html( $city ); ?></span>
                <?php endif; ?>
                <?php if ( $rating !== '' ) : ?>
                    <span class="rccc-pill">⭐ <?php echo esc_html( $rating ); ?><?php if ( $reviews !== '' ) : ?> (<?php echo esc_html( $reviews ); ?>)<?php endif; ?></span>
                <?php endif; ?>
            </div>
            <div class="rccc-actions">
                <a class="rccc-btn rccc-btn-primary" href="<?php echo esc_url( add_query_arg( array_filter( [ 'category' => $first_cat ?: null, 'zip' => $first_zip ?: null ] ), $lead_url ) ); ?>">Request a Quote</a>
                <a class="rccc-btn" href="<?php echo esc_url( get_post_type_archive_link( 'contractor' ) ); ?>">Back to Contractors</a>
            </div>
        </div>

        <?php if ( $cats ) : ?>
            <div class="rccc-line"><strong>Services:</strong>
                <?php foreach ( $cats as $c ) : ?>
                    <span class="rccc-chip"><?php echo esc_html( $c ); ?></span>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if ( $zips ) : ?>
            <div class="rccc-line"><strong>Service ZIPs:</strong> <?php echo esc_html( $zips ); ?></div>
        <?php endif; ?>

        <?php echo $desc; ?>

        <div class="rccc-meta">
            <?php if ( $license ) : ?>
                <div>License: <?php echo esc_html( $license ); ?></div>
            <?php endif; ?>
            <?php if ( $insExp ) : ?>
                <div>Insurance expires: <?php echo esc_html( $insExp ); ?></div>
            <?php endif; ?>
            <?php if ( $email ) : ?>
                <div>Email: <a href="mailto:<?php echo esc_attr( $email ); ?>"><?php echo esc_html( $email ); ?></a></div>
            <?php endif; ?>
        </div>

        <?php
        // Reviews block
        $avg_rating  = get_post_meta( $id, 'rccc_rating', true );
        $num_reviews = get_post_meta( $id, 'rccc_reviews', true );
        ?>
        <section class="rccc-reviews" id="rccc-reviews">
            <h3>Reviews</h3>
            <div class="rccc-avg">
                <?php
                if ( $avg_rating ) {
                    echo rccc_star_html( $avg_rating );
                    echo '<span>' . esc_html( $avg_rating ) . ' (' . intval( $num_reviews ) . ' reviews)</span>';
                } else {
                    echo '<span>No ratings yet. Be the first to leave one.</span>';
                }
                ?>
            </div>

            <?php
            $comments = get_comments( [
                'post_id'    => $id,
                'status'     => 'approve',
                'type'       => 'comment',
                'meta_query' => [
                    [
                        'key'     => 'rccc_rating',
                        'compare' => 'EXISTS',
                    ],
                ],
            ] );

            if ( $comments ) {
                wp_list_comments( [
                    'callback'    => 'rccc_review_item_cb',
                    'style'       => 'div',
                    'avatar_size' => 0,
                    'max_depth'   => 1,
                ], $comments );
            }
            ?>

            <?php comment_form(); ?>
        </section>
    </div>
    <?php
    return ob_get_clean();
}
add_filter( 'the_content', 'rccc_render_single_contractor_card', 9 );

/* ---------- Reviews meta helpers ---------- */

function rccc_star_html( $value ) {
    $value   = floatval( $value );
    $rounded = round( $value );
    $out     = '<span class="rccc-stars" aria-label="' . esc_attr( $value ) . ' out of 5">';
    for ( $i = 1; $i <= 5; $i++ ) {
        $out .= '<span class="rccc-star' . ( $i <= $rounded ? ' is-on' : '' ) . '">★</span>';
    }
    return $out . '</span>';
}

function rccc_review_item_cb( $comment, $args, $depth ) {
    $rating = (int) get_comment_meta( $comment->comment_ID, 'rccc_rating', true );
    echo '<div class="rccc-review" id="comment-' . intval( $comment->comment_ID ) . '">';
    echo   '<div class="rccc-review-head"><strong>' . esc_html( get_comment_author( $comment ) ) . '</strong>';
    echo   ' <span class="rccc-review-date">' . esc_html( get_comment_date( '', $comment ) ) . '</span>';
    if ( $rating ) echo rccc_star_html( $rating );
    echo   '</div>';
    // Allow a subset of HTML in reviews (links allowed)
    echo   '<div class="rccc-review-body">' . wpautop( wp_kses_post( get_comment_text( $comment ) ) ) . '</div>';
    echo '</div>';
}

/* === Contractor Submit Shortcode + ACF boot === */

add_action( 'wp', function () {
    if ( ! function_exists( 'acf_form_head' ) ) return;
    if ( ! is_singular() ) return;
    global $post;
    if ( $post && has_shortcode( $post->post_content, 'contractor_submit' ) ) {
        acf_form_head();
    }
} );

function rccc_render_contractor_submit_shortcode() {
    if ( ! function_exists( 'acf_form' ) ) {
        return '<p>ACF is not active.</p>';
    }

    // Must be logged in
    if ( ! is_user_logged_in() ) {
        $reg_url   = esc_url( home_url( '/contractor-registration/' ) );
        $login_url = esc_url( home_url( '/contractor-login/' ) );

        return '<p>You must be logged in to submit or edit your contractor profile.</p>'
             . '<p><a href="' . $reg_url . '">Register</a> or '
             . '<a href="' . $login_url . '">log in</a>.</p>';
    }

    $current_user_id = get_current_user_id();

    // Existing contractor post owned by this user
    $existing = get_posts( [
        'post_type'      => 'contractor',
        'author'         => $current_user_id,
        'post_status'    => [ 'publish', 'pending', 'draft' ],
        'posts_per_page' => 1,
        'fields'         => 'ids',
    ] );

    $is_edit = ! empty( $existing );
    $post_id = $is_edit ? $existing[0] : 'new_post';

    $form_args = [
        'post_title'   => true,
        'post_content' => false,
        'uploader'     => 'wp',
        'return'       => home_url( '/contractor-welcome/' ),
        'submit_value' => $is_edit ? 'Update Contractor' : 'Submit Contractor',
    ];

    if ( $is_edit ) {
        $form_args['post_id'] = $post_id;
    } else {
        $form_args['post_id']  = 'new_post';
        $form_args['new_post'] = [
            'post_type'   => 'contractor',
            'post_status' => 'publish',
        ];
    }

    ob_start(); ?>
    <section class="rccc-submit">
        <div class="rccc-hero">
            <h1>Join RainCity Contractor Connect</h1>
            <p>Tell us about your company, coverage area, credentials, and show recent projects. It takes about 3–5 minutes.</p>
            <ul class="rccc-hero-bullets">
                <li>✔ Appears in homeowner searches by ZIP &amp; service</li>
                <li>✔ You control your profile &amp; photos</li>
                <li>✔ Leads routed instantly via email</li>
            </ul>
        </div>

        <div class="rccc-wizard">
            <div class="rccc-progress">
                <div class="rccc-progress-bar"><span style="width:0%"></span></div>
                <div class="rccc-steps"></div>
            </div>

            <div class="rccc-form-wrap">
                <?php acf_form( $form_args ); ?>
                <p>If you are not redirected automatically, <a href="<?php echo esc_url( home_url( '/contractor-welcome/' ) ); ?>">click here to view your welcome page</a>.</p>
            </div>

            <div class="rccc-nav">
                <button type="button" class="rccc-btn rccc-btn-secondary" data-rccc-prev disabled>Back</button>
                <button type="button" class="rccc-btn rccc-btn-primary" data-rccc-next>Next</button>
                <div class="rccc-note">You can review before submitting.</div>
            </div>

            <div class="rccc-legal">
                <label class="rccc-check">
                    <input type="checkbox" id="rccc-terms">
                    <span>I confirm the information is accurate and I agree to be contacted about leads.</span>
                </label>
            </div>
        </div>
    </section>
    <?php
    return ob_get_clean();
}
add_shortcode( 'contractor_submit', 'rccc_render_contractor_submit_shortcode' );

/* === Contractor Registration via WPForms (form ID 241) === */
add_action( 'wpforms_process_complete_241', 'rccc_register_contractor_from_wpforms', 20, 4 );

function rccc_register_contractor_from_wpforms( $fields, $entry, $form_data, $entry_id ) {
    if ( ! empty( wpforms()->process->errors[ $form_data['id'] ] ) ) {
        return;
    }

    $company  = '';
    $email    = '';
    $password = '';

    foreach ( $fields as $f ) {
        $name = strtolower( trim( $f['name'] ?? '' ) );
        $val  = $f['value'] ?? '';

        if ( $name === 'company name' || $name === 'company' ) {
            $company = sanitize_text_field( $val );
        } elseif ( $name === 'email' || $name === 'email address' ) {
            $email = sanitize_email( $val );
        } elseif ( $name === 'password' ) {
            $password = (string) $val;
        }
    }

    if ( ! $email ) {
        foreach ( $fields as $f ) {
            $val = $f['value'] ?? '';
            if ( is_string( $val ) && filter_var( $val, FILTER_VALIDATE_EMAIL ) ) {
                $email = sanitize_email( $val );
                break;
            }
        }
    }

    if ( ! $email || ! is_email( $email ) ) {
        return;
    }

    if ( ! $password ) {
        $password = wp_generate_password( 12, false );
    }

    $user = get_user_by( 'email', $email );

    if ( $user ) {
        $creds = [
            'user_login'    => $user->user_login,
            'user_password' => $password,
            'remember'      => true,
        ];
        $signon = wp_signon( $creds, false );
        if ( ! is_wp_error( $signon ) ) {
            wp_set_current_user( $signon->ID );
            wp_set_auth_cookie( $signon->ID );
        }
    } else {
        $base_username = sanitize_user( current( explode( '@', $email ) ), true );
        $username      = $base_username ?: 'contractor';

        $i = 1;
        while ( username_exists( $username ) ) {
            $username = $base_username . '_' . $i;
            $i++;
        }

        $user_id = wp_create_user( $username, $password, $email );

        if ( ! is_wp_error( $user_id ) ) {
            $update = [
                'ID'   => $user_id,
                'role' => 'author',
            ];
            if ( ! empty( $company ) ) {
                $update['display_name'] = $company;
            }
            wp_update_user( $update );

            wp_set_current_user( $user_id );
            wp_set_auth_cookie( $user_id );
        }
    }

    $redirect = home_url( RCCC_REG_REDIRECT_PATH );
    wp_safe_redirect( $redirect );
    exit;
}

/* ---------- Reviews: comments + stars (validation & meta updates) ---------- */

add_filter( 'comment_form_defaults', function ( $d ) {
    if ( ! is_singular( 'contractor' ) ) return $d;
    $rating = '
    <p class="comment-form-rating">
        <label for="rccc_rating">Your rating <span class="required">*</span></label>
        <select name="rccc_rating" id="rccc_rating" required>
            <option value="">Rate…</option>
            <option value="5">★★★★★ (5)</option>
            <option value="4">★★★★☆ (4)</option>
            <option value="3">★★★☆☆ (3)</option>
            <option value="2">★★☆☆☆ (2)</option>
            <option value="1">★☆☆☆☆ (1)</option>
        </select>
    </p>';
    $d['comment_field'] = $rating . $d['comment_field'];
    $d['label_submit']  = 'Submit Review';
    $d['title_reply']   = 'Leave a Review';
    return $d;
} );

add_filter( 'preprocess_comment', function ( $c ) {
    if ( get_post_type( $c['comment_post_ID'] ) === 'contractor' ) {
        $ok = isset( $_POST['rccc_rating'] ) && in_array( (int) $_POST['rccc_rating'], [ 1,2,3,4,5 ], true );
        if ( ! $ok ) wp_die( 'Please provide a star rating (1–5).' );
    }
    return $c;
} );

function rccc_recalc_rating( $post_id ) {
    $comments = get_comments( [
        'post_id'    => $post_id,
        'status'     => 'approve',
        'type'       => 'comment',
        'meta_query' => [ [ 'key' => 'rccc_rating', 'compare' => 'EXISTS' ] ],
    ] );
    $sum   = 0;
    $count = 0;
    foreach ( $comments as $cm ) {
        $r = (int) get_comment_meta( $cm->comment_ID, 'rccc_rating', true );
        if ( $r >= 1 && $r <= 5 ) {
            $sum   += $r;
            $count++;
        }
    }
    if ( $count > 0 ) {
        update_post_meta( $post_id, 'rccc_rating',  round( $sum / $count, 1 ) );
        update_post_meta( $post_id, 'rccc_reviews', $count );
    } else {
        delete_post_meta( $post_id, 'rccc_rating' );
        delete_post_meta( $post_id, 'rccc_reviews' );
    }
}

add_action( 'comment_post', function ( $comment_id, $approved ) {
    $comment = get_comment( $comment_id );
    if ( ! $comment || get_post_type( $comment->comment_post_ID ) !== 'contractor' ) return;
    $rating = isset( $_POST['rccc_rating'] ) ? (int) $_POST['rccc_rating'] : 0;
    if ( $rating >= 1 && $rating <= 5 ) update_comment_meta( $comment_id, 'rccc_rating', $rating );
    if ( (int) $approved === 1 ) rccc_recalc_rating( $comment->comment_post_ID );
}, 10, 2 );

add_action( 'transition_comment_status', function ( $new, $old, $comment ) {
    if ( $comment && get_post_type( $comment->comment_post_ID ) === 'contractor' ) rccc_recalc_rating( $comment->comment_post_ID );
}, 10, 3 );

add_action( 'edit_comment', function ( $comment_id ) {
    $comment = get_comment( $comment_id );
    if ( $comment && get_post_type( $comment->comment_post_ID ) === 'contractor' ) rccc_recalc_rating( $comment->comment_post_ID );
} );

add_action( 'deleted_comment', function ( $comment_id ) {
    $comment = get_comment( $comment_id );
    if ( $comment && get_post_type( $comment->comment_post_ID ) === 'contractor' ) rccc_recalc_rating( $comment->comment_post_ID );
} );

/* === FRONT-END CONTRACTOR LOGIN SHORTCODE === */

function rccc_render_contractor_login_shortcode( $atts = [] ) {
    $atts = shortcode_atts(
        [ 'redirect' => '/submit-contractor/' ],
        $atts,
        'contractor_login'
    );

    $redirect_url = esc_url_raw( home_url( $atts['redirect'] ) );
    $errors       = [];

    if ( is_user_logged_in() && ! current_user_can( 'manage_options' ) ) {
        wp_safe_redirect( $redirect_url );
        exit;
    }

    if (
        $_SERVER['REQUEST_METHOD'] === 'POST'
        && isset( $_POST['rccc_login_action'] )
        && $_POST['rccc_login_action'] === 'login'
    ) {
        check_admin_referer( 'rccc_login', 'rccc_login_nonce' );

        $login    = isset( $_POST['rccc_login'] ) ? sanitize_text_field( wp_unslash( $_POST['rccc_login'] ) ) : '';
        $password = isset( $_POST['rccc_password'] ) ? (string) $_POST['rccc_password'] : '';

        if ( $login === '' || $password === '' ) {
            $errors[] = 'Please enter your email (or username) and password.';
        } else {
            $creds = [
                'user_login'    => $login,
                'user_password' => $password,
                'remember'      => ! empty( $_POST['rccc_remember'] ),
            ];

            $user = wp_signon( $creds, false );

            if ( is_wp_error( $user ) ) {
                $errors[] = $user->get_error_message();
            } else {
                if ( user_can( $user, 'manage_options' ) ) {
                    wp_safe_redirect( admin_url() );
                } else {
                    wp_safe_redirect( $redirect_url );
                }
                exit;
            }
        }
    }

    ob_start();
    ?>
    <section class="rccc-login">
        <h1>Contractor Login</h1>
        <p>Log in to edit your company profile and manage your information.</p>

        <?php if ( ! empty( $errors ) ) : ?>
            <div class="rccc-login-errors">
                <ul>
                    <?php foreach ( $errors as $e ) : ?>
                        <li><?php echo wp_kses_post( $e ); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <form method="post">
            <?php wp_nonce_field( 'rccc_login', 'rccc_login_nonce' ); ?>
            <input type="hidden" name="rccc_login_action" value="login">

            <p>
                <label for="rccc_login">Email (or Username)</label><br>
                <input type="text" id="rccc_login" name="rccc_login" required>
            </p>

            <p>
                <label for="rccc_password">Password</label><br>
                <input type="password" id="rccc_password" name="rccc_password" required>
            </p>

            <p>
                <label>
                    <input type="checkbox" name="rccc_remember" value="1">
                    Remember me
                </label>
            </p>

            <p>
                <button type="submit" class="rccc-btn rccc-btn-primary">Log In</button>
            </p>
        </form>
    </section>
    <?php
    return ob_get_clean();
}
add_shortcode( 'contractor_login', 'rccc_render_contractor_login_shortcode' );

/* === Contractor role helper + redirects (login + wp-admin) === */

function rccc_is_contractor_user( $user = null ) {
    if ( $user instanceof WP_User ) {
        $roles = (array) $user->roles;
    } else {
        $user = wp_get_current_user();
        if ( ! $user || ! $user->ID ) {
            return false;
        }
        $roles = (array) $user->roles;
    }

    $contractor_roles = [ 'author' ];
    return (bool) array_intersect( $contractor_roles, $roles );
}

function rccc_login_redirect( $redirect_to, $requested_redirect_to, $user ) {
    if ( is_wp_error( $user ) || ! ( $user instanceof WP_User ) ) {
        return $redirect_to;
    }

    if ( in_array( 'administrator', (array) $user->roles, true ) ) {
        return $redirect_to ?: admin_url();
    }

    return home_url( '/submit-contractor/' );
}
add_filter( 'login_redirect', 'rccc_login_redirect', 10, 3 );

add_action( 'admin_init', function () {
    if ( ! is_user_logged_in() ) {
        return;
    }

    if ( current_user_can( 'manage_options' ) || current_user_can( 'edit_theme_options' ) || wp_doing_ajax() ) {
        return;
    }

    if ( rccc_is_contractor_user() ) {
        wp_safe_redirect( home_url( '/submit-contractor/' ) );
        exit;
    }
} );

/* === Contractor Welcome shortcode (uses latest contractor for current user) === */

if ( ! function_exists( 'rccc_get_current_contractor' ) ) {
    function rccc_get_current_contractor() {
        if ( ! is_user_logged_in() ) {
            return null;
        }

        $user_id = get_current_user_id();

        $posts = get_posts( array(
            'post_type'      => 'contractor',
            'author'         => $user_id,
            'post_status'    => array( 'publish', 'pending', 'draft' ),
            'posts_per_page' => 1,
            'orderby'        => 'date',
            'order'          => 'DESC',
        ) );

        if ( empty( $posts ) ) {
            return null;
        }

        return $posts[0]; // WP_Post
    }
}


if ( ! function_exists( 'rccc_contractor_welcome_shortcode' ) ) {
    function rccc_contractor_welcome_shortcode( $atts = array() ) {

        $contractor = rccc_get_current_contractor();
        $logo_url   = '';
        $name       = '';

        if ( $contractor instanceof WP_Post ) {
            $name = get_the_title( $contractor->ID );

            // Try ACF logo first
            if ( function_exists( 'get_field' ) ) {
                $acf_logo = get_field( 'rccc_contractor_logo', $contractor->ID );
                if ( is_array( $acf_logo ) && ! empty( $acf_logo['ID'] ) ) {
                    $logo_url = wp_get_attachment_image_url( (int) $acf_logo['ID'], 'medium' );
                } elseif ( is_numeric( $acf_logo ) ) {
                    $logo_url = wp_get_attachment_image_url( (int) $acf_logo, 'medium' );
                } elseif ( is_string( $acf_logo ) && filter_var( $acf_logo, FILTER_VALIDATE_URL ) ) {
                    $logo_url = $acf_logo;
                }
            }

            // Fallback to featured image
            if ( ! $logo_url ) {
                $thumb_id = get_post_thumbnail_id( $contractor->ID );
                if ( $thumb_id ) {
                    $logo_url = wp_get_attachment_image_url( $thumb_id, 'medium' );
                }
            }
        }

        ob_start(); ?>
        <div class="welcome-container">
            <h1 class="page-title">Welcome to RainCity Contractor Connect</h1>
            <p>Your contractor profile has been submitted successfully. We’ll use your company details to match you with local homeowners.</p>

            <?php if ( $contractor instanceof WP_Post ) : ?>
                <?php if ( $logo_url ) : ?>
                    <img src="<?php echo esc_url( $logo_url ); ?>" alt="<?php echo esc_attr( $name ); ?> logo" class="contractor-logo">
                <?php else : ?>
                    <div class="contractor-logo" style="background:#047857;color:#fff;display:flex;align-items:center;justify-content:center;font-size:36px;">
                        <?php echo esc_html( mb_strtoupper( mb_substr( $name, 0, 1 ) ) ); ?>
                    </div>
                <?php endif; ?>

                <div class="contractor-details">
                    <p><strong><?php echo esc_html( $name ); ?></strong></p>
                    <p>You can edit your details, photos, and coverage area at any time.</p>
                </div>
            <?php else : ?>
                <p>No contractor profile found for your account.</p>
            <?php endif; ?>

            <a href="<?php echo esc_url( home_url( '/contractors/' ) ); ?>" class="button">View Contractor Directory</a>
            <a href="<?php echo esc_url( home_url( '/submit-contractor/' ) ); ?>" class="button">Edit Your Profile</a>
        </div>
        <?php
        return ob_get_clean();
    }
}



/* ---------- Styles & JS ---------- */
add_action( 'wp_enqueue_scripts', function () {

    // Single contractor styles
    if ( is_singular( 'contractor' ) ) {
        $css = '
        .rccc-container{max-width:1180px;margin:0 auto;padding:0 16px}
        .rccc-layout{display:grid;grid-template-columns:minmax(0,740px) 320px;gap:24px;align-items:flex-start}
        @media (min-width:1024px){.rccc-aside{position:sticky;top:90px}}
        @media (max-width:1000px){
            .rccc-layout{grid-template-columns:1fr}
            .rccc-aside{position:static;margin-top:16px}
        }

        body.single-contractor .entry-title{display:none}
        body.single-contractor .post-thumbnail,
        body.single-contractor .entry-thumbnail,
        body.single-contractor .wp-post-image{display:none !important;}

        .rccc-single-card{background:#fff;border:1px solid #e5e7eb;border-radius:16px;padding:24px;box-shadow:0 4px 14px rgba(0,0,0,.06)}
        .rccc-single-card>*+*{margin-top:12px}
        .rccc-single-card p{max-width:70ch;line-height:1.6}

        .rccc-header{display:flex;align-items:center;gap:12px;margin-bottom:6px}
        .rccc-logo{width:80px;height:80px;border-radius:12px;border:1px solid #e5e7eb;background:#fff;object-fit:contain;padding:6px}
        .rccc-logo--placeholder{display:flex;align-items:center;justify-content:center;font-weight:700;color:#475569}
        .rccc-name{margin:0;font-size:clamp(20px,2.2vw,24px);line-height:1.2;color:#0f172a}

        .rccc-top{display:flex;justify-content:space-between;gap:12px;flex-wrap:wrap;margin-bottom:8px}
        .rccc-badges{display:flex;flex-wrap:wrap;gap:6px}
        .rccc-pill{display:inline-block;background:#eef2ff;color:#3730a3;border-radius:999px;padding:4px 10px;font-size:12px}
        .rccc-actions{display:flex;flex-wrap:wrap;gap:8px}

        .rccc-btn{display:inline-block;background:#0f172a;color:#fff;text-decoration:none;border-radius:10px;padding:8px 12px;font-size:13px;border:none;cursor:pointer}
        .rccc-btn-primary{background:#047857}

        .rccc-chip{display:inline-block;background:#f1f5f9;border:1px solid #e5e7eb;border-radius:999px;padding:3px 10px;font-size:12px;margin:0 6px 6px 0}
        .rccc-line{margin:6px 0;color:#334155}
        .rccc-meta{margin-top:12px;color:#475569;font-size:14px}

        .rccc-reviews{margin-top:24px;padding:16px;border:1px solid #e5e7eb;border-radius:12px;background:#fff}
        .rccc-avg{display:flex;align-items:center;gap:8px;margin-bottom:12px;color:#334155}
        .rccc-stars .rccc-star{font-size:18px;color:#cbd5e1;margin-right:2px}
        .rccc-stars .rccc-star.is-on{color:#f59e0b}
        .rccc-review{border-top:1px solid #f1f5f9;padding-top:10px;margin-top:10px}
        .rccc-review-head{display:flex;align-items:center;gap:8px;color:#0f172a}
        .rccc-review-date{color:#64748b;font-size:12px}
        .comment-form-rating select{border:1px solid #e2e8f0;border-radius:8px;padding:6px}

        body.single-contractor .comments-area,
        body.single-contractor .entry-comments,
        body.single-contractor .post-comments,
        body.single-contractor .comments,
        body.single-contractor #comments{
            display:none !important;
        }

        .rccc-projects{margin:0 0 16px}
        .rccc-project{border:1px solid #e5e7eb;border-radius:12px;padding:12px;margin-bottom:12px;background:#fff}
        .rccc-project h4{margin:0 0 8px;font-size:14px}
        .rccc-grid-2{display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:6px}
        .rccc-thumb{width:100%;aspect-ratio:4/3;border-radius:10px;overflow:hidden;background:#f8fafc}
        .rccc-thumb img{width:100%;height:100%;object-fit:cover;display:block}

        .rccc-ad{background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:12px;margin-bottom:12px}
        .rccc-ad--ph{display:flex;align-items:center;justify-content:center;color:#64748b;background:#f8fafc}
        .rccc-ad--tall{min-height:600px}
        ';
        wp_register_style( 'rccc-single-css', false );
        wp_enqueue_style( 'rccc-single-css' );
        wp_add_inline_style( 'rccc-single-css', $css );
    }

    // Front-end ACF submit page styles + wizard JS
    if ( is_singular() ) {
        global $post;
        if ( $post && strpos( $post->post_content, '[contractor_submit]' ) !== false ) {

            $css = '
            .rccc-submit{max-width:980px;margin:24px auto;padding:0 16px}
            .rccc-hero{background:#0f172a;color:#e5e7eb;border-radius:16px;padding:18px 20px;margin-bottom:16px}
            .rccc-hero h1{margin:0 0 6px;color:#fff;font-size:26px}
            .rccc-hero-bullets{margin:8px 0 0 16px;padding:0}
            .rccc-hero-bullets li{list-style:disc}
            .rccc-progress{margin:12px 0}
            .rccc-progress-bar{height:8px;background:#e5e7eb;border-radius:999px;overflow:hidden}
            .rccc-progress-bar span{display:block;height:100%;background:#10b981}
            .rccc-steps{display:flex;flex-wrap:wrap;gap:6px;margin-top:8px}
            .rccc-step-chip{font-size:12px;padding:4px 10px;border-radius:999px;background:#f1f5f9;border:1px solid #e2e8f0}
            .rccc-step-chip.is-active{background:#d1fae5;border-color:#34d399}
            .rccc-form-wrap{background:#fff;border:1px solid #e5e7eb;border-radius:16px;padding:16px}
            .acf-form .acf-fields>.acf-field{padding:12px 0;border-top:1px solid #f1f5f9}
            .acf-form .acf-label label{font-weight:600;color:#0f172a}
            .acf-form input[type=text],.acf-form input[type=email],.acf-form input[type=number],.acf-form .acf-input select,.acf-form .acf-input textarea{
                width:100%;border:1px solid #e2e8f0;border-radius:10px;padding:10px 12px;box-shadow:none;outline:none
            }
            .acf-form input:focus,.acf-form select:focus,.acf-form textarea:focus{border-color:#10b981;box-shadow:0 0 0 3px rgba(16,185,129,.15)}
            .acf-form .acf-gallery{border-radius:12px;border-color:#e2e8f0}
            .acf-form .acf-button.button{background:#10b981;border:none;border-radius:10px;padding:10px 16px}
            .acf-form .acf-button.button:hover{background:#0ea371}
            .rccc-nav{display:flex;gap:10px;align-items:center;margin:12px 0}
            .rccc-btn{display:inline-block;background:#0f172a;color:#fff;border-radius:10px;padding:10px 14px;text-decoration:none;border:none;cursor:pointer}
            .rccc-btn-secondary{background:#475569}
            .rccc-btn[disabled]{opacity:.55;cursor:not-allowed}
            .rccc-legal{margin-top:6px;color:#334155}
            .rccc-check{display:flex;gap:8px;align-items:center}
            ';
            wp_register_style( 'rccc-front-form', false );
            wp_enqueue_style( 'rccc-front-form' );
            wp_add_inline_style( 'rccc-front-form', $css );

            wp_register_script( 'rccc-submit-js', '', [], null, true );
            wp_enqueue_script( 'rccc-submit-js' );
            $js = <<<JS
            (function($){
                function initWizard(){
                    var \$wrap = $('.rccc-submit');
                    if (!\$wrap.length) return;

                    var \$form = \$wrap.find('.acf-form');
                    if (!\$form.length) return;

                    var \$buttons = \$form.find('.acf-tab-wrap .acf-tab-button');
                    if (!\$buttons.length) return;

                    var \$chips = \$wrap.find('.rccc-steps').empty();
                    \$buttons.each(function(i){
                        var label = $(this).text().trim() || ('Step '+(i+1));
                        $('<span class="rccc-step-chip"/>')
                            .text((i+1)+'. '+label)
                            .appendTo(\$chips);
                    });

                    var current = 0;

                    function updateUI(){
                        var total = \$buttons.length;
                        var pct   = ((current+1)/total)*100;

                        \$wrap.find('.rccc-progress-bar span').css('width', pct+'%');

                        \$chips.children().removeClass('is-active').eq(current).addClass('is-active');

                        \$wrap.find('[data-rccc-prev]').prop('disabled', current === 0);
                        var isLast = (current === total - 1);
                        \$wrap.find('[data-rccc-next]').toggle(!isLast);

                        var \$submitWrap = \$form.find('.acf-form-submit');
                        if (\$submitWrap.length){
                            if (isLast){
                                \$submitWrap.show();
                            } else {
                                \$submitWrap.hide();
                            }
                        }
                    }

                    function goTo(index){
                        var total = \$buttons.length;
                        if (index < 0 || index >= total) return;
                        current = index;
                        \$buttons.eq(current).trigger('click');
                        updateUI();
                    }

                    \$wrap.on('click','[data-rccc-prev]', function(){
                        goTo(current - 1);
                    });

                    \$wrap.on('click','[data-rccc-next]', function(){
                        goTo(current + 1);
                    });

                    \$buttons.on('click', function(){
                        current = \$buttons.index(this);
                        updateUI();
                    });

                    \$form.on('submit', function(e){
                        var isLast = (current === \$buttons.length - 1);
                        if (!isLast){
                            e.preventDefault();
                            goTo(current + 1);
                            return;
                        }
                        var terms = document.getElementById('rccc-terms');
                        if (terms && !terms.checked){
                            e.preventDefault();
                            alert('Please agree to the confirmation before submitting.');
                        }
                    });

                    goTo(0);
                }

                if (window.acf && acf.add_action){
                    acf.add_action('ready', initWizard);
                    acf.add_action('append', initWizard);
                } else {
                    $(initWizard);
                }
            })(jQuery);
JS;
            wp_add_inline_script( 'rccc-submit-js', $js );
        }
    }
} );



// Optional: ACF block registration for welcome block (safe even if never used)
add_action( 'acf/init', function () {
    if ( function_exists( 'acf_register_block_type' ) ) {
        acf_register_block_type( [
            'name'            => 'contractor-welcome',
            'title'           => __( 'Contractor Welcome', 'rccc' ),
            'description'     => __( 'Displays a success message after contractor submission.', 'rccc' ),
            'render_callback' => 'rccc_contractor_welcome_shortcode',
            'category'        => 'widgets',
            'icon'            => 'smiley',
            'keywords'        => [ 'contractor', 'welcome', 'success' ],
            'supports'        => [
                'align' => true,
                'mode'  => false,
            ],
        ] );
    }
} );
