<?php
/**
 * Plugin Name: RCCC Contractor Role
 * Description: Adds a lightweight "contractor" role and (optionally) migrates users who own a contractor post to that role. Safe to activate on staging; review results and then keep active or deactivate.
 * Version: 1.0.0
 * Author: RCCC
 *
 * Install:
 * 1. Create folder: wp-content/plugins/rccc-contractor-role/
 * 2. Save this file as wp-content/plugins/rccc-contractor-role/rccc-contractor-role.php
 * 3. Activate the plugin (Plugins -> Installed Plugins).
 *
 * What it does:
 * - On activation: creates a "contractor" role with a small set of capabilities and
 *   migrates users who are the author of a "contractor" post to the contractor role
 *   (and removes their 'author' role).
 * - It does NOT delete users. It does NOT remove the role on deactivation.
 *
 * Note:
 * - After activating, update your registration flow to assign 'contractor' as the role
 *   for new contractor signups (example below).
 * - If you prefer a different capability set, edit the $caps array below before activating.
 */

/* Bail if called directly. */
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Desired role key.
 */
define( 'RCCC_CONTRACTOR_ROLE', 'contractor' );

/**
 * Capabilities for the Contractor role.
 * Adjust as needed. These are intentionally minimal.
 */
function rccc_contractor_caps() {
    return array(
        'read'                    => true,  // allow dashboard access
        'upload_files'            => true,  // allow logo/media uploads
        'edit_posts'              => true,  // allows editing (useful for ACF front-end)
        'edit_published_posts'    => true,
        'publish_posts'           => false, // prevent publishing standard blog posts
        'delete_posts'            => false,
    );
}

/**
 * Activation: add role and migrate existing contractor authors.
 */
function rccc_contractor_role_activate() {
    // Add the role if it doesn't exist
    if ( ! get_role( RCCC_CONTRACTOR_ROLE ) ) {
        add_role( RCCC_CONTRACTOR_ROLE, 'Contractor', rccc_contractor_caps() );
    } else {
        // ensure caps are set (update if needed)
        $role = get_role( RCCC_CONTRACTOR_ROLE );
        if ( $role ) {
            foreach ( rccc_contractor_caps() as $cap => $grant ) {
                if ( $grant ) {
                    $role->add_cap( $cap );
                } else {
                    $role->remove_cap( $cap );
                }
            }
        }
    }

    // Migrate existing users who authored a contractor post to the new role.
    // Finds contractor posts and moves their authors (if any) to contractor role.
    $posts = get_posts( array(
        'post_type'      => 'contractor',
        'posts_per_page' => -1,
        'fields'         => 'ids',
        'post_status'    => array( 'publish', 'pending', 'draft' ),
    ) );

    if ( ! empty( $posts ) ) {
        $author_ids = array();
        foreach ( $posts as $pid ) {
            $post = get_post( $pid );
            if ( $post && ! empty( $post->post_author ) ) {
                $author_ids[] = (int) $post->post_author;
            }
        }
        $author_ids = array_unique( $author_ids );

        $migrated = 0;
        foreach ( $author_ids as $uid ) {
            $user = get_user_by( 'ID', $uid );
            if ( ! $user ) continue;

            // Add role 'contractor' if not present
            if ( ! in_array( RCCC_CONTRACTOR_ROLE, (array) $user->roles, true ) ) {
                $user->add_role( RCCC_CONTRACTOR_ROLE );
                $migrated++;
            }

            // Optionally remove 'author' role if present
            if ( in_array( 'author', (array) $user->roles, true ) ) {
                $user->remove_role( 'author' );
            }
        }

        // Store a transient to show an admin notice summarizing the migration once
        set_transient( 'rccc_contractor_role_migrated_count', $migrated, 60 );
    }
}
register_activation_hook( __FILE__, 'rccc_contractor_role_activate' );

/**
 * Admin notice after activation showing how many users were migrated.
 */
function rccc_contractor_role_admin_notice() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    $migrated = get_transient( 'rccc_contractor_role_migrated_count' );
    if ( $migrated !== false ) {
        delete_transient( 'rccc_contractor_role_migrated_count' );
        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php echo esc_html( sprintf( 'RCCC Contractor Role: migrated %d user(s) to the "%s" role.', (int) $migrated, RCCC_CONTRACTOR_ROLE ) ); ?></p>
        </div>
        <?php
    }
}
add_action( 'admin_notices', 'rccc_contractor_role_admin_notice' );

/**
 * Helper: assign contractor role when creating users programmatically.
 * Example usage: when registering a new user, set role => 'contractor' in wp_create_user or wp_update_user.
 *
 * Example change to your registration code:
 *  Replace 'role' => 'author' with 'role' => 'contractor'
 *
 * Before:
 *   $user_id = wp_create_user( $username, $password, $email );
 *   if ( ! is_wp_error( $user_id ) ) {
 *       wp_update_user( [ 'ID' => $user_id, 'role' => 'author' ] );
 *   }
 *
 * After:
 *   $user_id = wp_create_user( $username, $password, $email );
 *   if ( ! is_wp_error( $user_id ) ) {
 *       wp_update_user( [ 'ID' => $user_id, 'role' => 'contractor' ] );
 *   }
 */

/**
 * Optional: deactivation cleanup.
 * NOTE: We intentionally do NOT remove the role on deactivation, because removing
 * roles may leave users without roles and is destructive. If you want to remove the
 * role, do it manually or modify this function.
 */
function rccc_contractor_role_deactivate() {
    // No destructive cleanup by default.
}
register_deactivation_hook( __FILE__, 'rccc_contractor_role_deactivate' );