<?php
/**
 * Plugin Name: RCCC Email Test
 * Description: Small admin tool to test wp_mail and capture wp_mail_failed events for debugging outgoing email.
 * Version: 1.0.0
 * Author: RCCC
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/* Register admin menu */
add_action( 'admin_menu', function() {
    add_management_page( 'RCCC Email Test', 'RCCC Email Test', 'manage_options', 'rccc-email-test', 'rccc_email_test_page' );
} );

/* Capture last wp_mail_failed info in transient for admin display */
add_action( 'wp_mail_failed', function( $wp_error ) {
    // Store serialized error info for a short time
    set_transient( 'rccc_last_wp_mail_failed', wp_json_encode( $wp_error ), 300 );
}, 10 );

/* Admin page */
function rccc_email_test_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $notice = '';

    if ( isset( $_POST['rccc_email_test_send'] ) && check_admin_referer( 'rccc_email_test_nonce', 'rccc_email_test_nonce_field' ) ) {
        $to = sanitize_email( wp_unslash( $_POST['rccc_email_test_to'] ) );
        if ( ! is_email( $to ) ) {
            $notice = '<div class="notice notice-error"><p>Enter a valid email address.</p></div>';
        } else {
            $subject = 'RCCC Mail Test — ' . home_url();
            $body    = "This is a test message from " . home_url() . "\n\nTime: " . date_i18n( 'Y-m-d H:i:s' );
            $headers = [ 'Content-Type: text/plain; charset=UTF-8', 'From: ' . get_bloginfo( 'name' ) . ' <no-reply@' . preg_replace( '/^www\./', '', $_SERVER['HTTP_HOST'] ) . '>' ];

            // Attempt send and capture result
            $sent = wp_mail( $to, $subject, $body, $headers );

            if ( $sent ) {
                $notice = '<div class="notice notice-success"><p>wp_mail returned true — message accepted by WordPress for delivery. Check the destination inbox/spam. If nothing appears, configure SMTP.</p></div>';
            } else {
                $notice = '<div class="notice notice-error"><p>wp_mail returned false — message was NOT accepted. Check server mail settings / host mail restrictions.</p></div>';
            }

            // Retrieve any recent failure details
            $last_err = get_transient( 'rccc_last_wp_mail_failed' );
            if ( $last_err ) {
                $err = json_decode( $last_err, true );
                $notice .= '<div style="margin-top:10px;padding:10px;background:#fff;border:1px solid #eee;"><strong>Last wp_mail_failed object (raw):</strong><pre style="white-space:pre-wrap">' . esc_html( print_r( $err, true ) ) . '</pre></div>';
                delete_transient( 'rccc_last_wp_mail_failed' );
            } else {
                $notice .= '<div style="margin-top:10px;padding:10px;background:#fff;border:1px solid #eee;"><strong>No wp_mail_failed record found.</strong></div>';
            }
        }
    }

    // Admin UI
    echo '<div class="wrap"><h1>RCCC Email Test</h1>';
    echo $notice;
    echo '<form method="post">';
    wp_nonce_field( 'rccc_email_test_nonce', 'rccc_email_test_nonce_field' );
    echo '<table class="form-table"><tr><th><label for="rccc_email_test_to">Send test to</label></th><td><input id="rccc_email_test_to" name="rccc_email_test_to" type="email" value="' . esc_attr( get_option( 'admin_email' ) ) . '" class="regular-text" required></td></tr></table>';
    submit_button( 'Send Test Email', 'primary', 'rccc_email_test_send' );
    echo '</form>';

    echo '<h2>Quick next steps</h2>';
    echo '<ol><li>Install WP Mail Logging to record outgoing messages.</li><li>Install WP Mail SMTP and configure Mailgun/SendGrid/Amazon SES (preferred) or SMTP using a domain email.</li><li>Ensure SPF/DKIM are configured for your domain.</li></ol>';

    echo '</div>';
}