<?php
/**
 * Plugin Name: RCCC Registration & Welcome Helpers
 * Description: Ensures WPForms contractor registration signs users in as "contractor", auto-creates a minimal contractor post for the user (so the welcome page shows info immediately), forces ACF-created contractor posts to be owned by the logged-in user, and exposes a logout button/shortcode.
 * Version: 1.1.0
 * Author: RCCC
 *
 * Install:
 * - Save to wp-content/plugins/rccc-registration-flow/rccc-registration-flow.php (overwrite existing)
 * - Activate plugin in WP Admin -> Plugins
 *
 * Notes:
 * - Update RCCC_REG_FORM_ID if your registration form has a different ID.
 * - The plugin creates a minimal contractor CPT post on registration if none exists for the user.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/* --- Configuration: set the registration WPForms ID and redirect targets --- */
if ( ! defined( 'RCCC_REG_FORM_ID' ) ) define( 'RCCC_REG_FORM_ID', 241 ); // change if your registration form has a different ID
if ( ! defined( 'RCCC_SUBMIT_CONTRACTOR_PATH' ) ) define( 'RCCC_SUBMIT_CONTRACTOR_PATH', '/submit-contractor/' );
if ( ! defined( 'RCCC_WELCOME_PATH' ) ) define( 'RCCC_WELCOME_PATH', '/contractor-welcome/' );

/* --- WPForms handler: create or sign in user, give contractor role, ensure contractor post, redirect to ACF submit page --- */
add_action( 'wpforms_process_complete', 'rccc_handle_wpforms_registration', 10, 4 );
function rccc_handle_wpforms_registration( $fields, $entry, $form_data, $entry_id ) {
    if ( (int) $form_data['id'] !== (int) RCCC_REG_FORM_ID ) {
        return;
    }

    // If WPForms recorded errors for this submission, bail
    if ( function_exists( 'wpforms' ) && ! empty( wpforms()->process->errors[ $form_data['id'] ] ) ) {
        return;
    }

    $company  = '';
    $email    = '';
    $password = '';

    foreach ( $fields as $f ) {
        $name = strtolower( trim( $f['name'] ?? '' ) );
        $val  = $f['value'] ?? '';

        if ( $name === 'company name' || $name === 'company' ) {
            $company = sanitize_text_field( $val );
        } elseif ( $name === 'email' || $name === 'email address' ) {
            $email = sanitize_email( $val );
        } elseif ( $name === 'password' ) {
            $password = (string) $val;
        } elseif ( empty( $email ) && is_string( $val ) && filter_var( $val, FILTER_VALIDATE_EMAIL ) ) {
            $email = sanitize_email( $val );
        }
    }

    if ( ! $email || ! is_email( $email ) ) {
        return;
    }

    if ( ! $password ) {
        $password = wp_generate_password( 12, false );
    }

    $user_id = 0;
    $is_new_user = false;

    // Try to find existing user by email
    $user = get_user_by( 'email', $email );

    if ( $user ) {
        $user_id = $user->ID;
        // Ensure user has contractor role
        if ( ! in_array( 'contractor', (array) $user->roles, true ) ) {
            $user->add_role( 'contractor' );
        }

        // Sign-on: try wp_signon with username and provided password; fallback to set auth cookie
        $creds = [
            'user_login'    => $user->user_login,
            'user_password' => $password,
            'remember'      => true,
        ];
        $signon = wp_signon( $creds, false );
        if ( ! is_wp_error( $signon ) ) {
            wp_set_current_user( $signon->ID );
            wp_set_auth_cookie( $signon->ID );
        } else {
            // fallback: just set current user
            wp_set_current_user( $user_id );
            wp_set_auth_cookie( $user_id );
        }
    } else {
        // create new user; pick username from email before @
        $base_username = sanitize_user( current( explode( '@', $email ) ), true );
        $username      = $base_username ?: 'contractor';

        $i = 1;
        while ( username_exists( $username ) ) {
            $username = $base_username . '_' . $i;
            $i++;
        }

        $user_id = wp_create_user( $username, $password, $email );

        if ( is_wp_error( $user_id ) ) {
            // creation failed — bail
            return;
        }

        $is_new_user = true;

        // assign contractor role (ensure role exists)
        $u = new WP_User( $user_id );
        $u->set_role( 'contractor' );

        // optionally set display name to company
        if ( ! empty( $company ) ) {
            wp_update_user( [ 'ID' => $user_id, 'display_name' => sanitize_text_field( $company ) ] );
        }

        wp_set_current_user( $user_id );
        wp_set_auth_cookie( $user_id );
    }

    // Ensure a contractor post exists for this user (creates a minimal one if needed)
    rccc_ensure_minimal_contractor_post_for_user( $user_id, $company, $email );

    // Redirect to the ACF submit-contractor page (ensure this path exists)
    $redirect = home_url( RCCC_SUBMIT_CONTRACTOR_PATH );
    wp_safe_redirect( $redirect );
    exit;
}

/* --- Ensure minimal contractor post exists for user --- */
if ( ! function_exists( 'rccc_ensure_minimal_contractor_post_for_user' ) ) {
    function rccc_ensure_minimal_contractor_post_for_user( $user_id, $company = '', $contact_email = '' ) {
        $user_id = (int) $user_id;
        if ( $user_id <= 0 ) return null;

        // 1) Does a contractor post already exist authored by this user?
        $by_author = get_posts( array(
            'post_type' => 'contractor',
            'author' => $user_id,
            'post_status' => array( 'publish','pending','draft' ),
            'posts_per_page' => 1,
            'fields' => 'ids',
        ) );
        if ( ! empty( $by_author ) ) {
            $pid = (int) $by_author[0];
            // ensure contact email meta exists
            if ( $contact_email ) {
                $meta = get_post_meta( $pid, 'rccc_contact_email', true );
                if ( empty( $meta ) ) update_post_meta( $pid, 'rccc_contact_email', sanitize_email( $contact_email ) );
            }
            return get_post( $pid );
        }

        // 2) Any contractor post matching this email in meta? (in case author isn't set)
        if ( $contact_email ) {
            $by_email = get_posts( array(
                'post_type' => 'contractor',
                'post_status' => array( 'publish','pending','draft' ),
                'posts_per_page' => 1,
                'fields' => 'ids',
                'meta_query' => array(
                    array(
                        'key' => 'rccc_contact_email',
                        'value' => sanitize_email( $contact_email ),
                        'compare' => '=',
                    ),
                ),
            ) );
            if ( ! empty( $by_email ) ) {
                $pid = (int) $by_email[0];
                // If the post has no author, or different author, assign to this user
                $post = get_post( $pid );
                if ( $post && (int) $post->post_author !== $user_id ) {
                    wp_update_post( array( 'ID' => $pid, 'post_author' => $user_id ) );
                }
                return get_post( $pid );
            }
        }

        // 3) Not found: create a minimal contractor post for this user
        $title = $company ? sanitize_text_field( $company ) : ( 'Contractor ' . $user_id );
        $insert = array(
            'post_type' => 'contractor',
            'post_title' => wp_strip_all_tags( $title ),
            'post_status' => 'publish', // can be 'pending' if you prefer moderation
            'post_author' => $user_id,
            'post_content' => '',
        );
        $pid = wp_insert_post( $insert, true );
        if ( is_wp_error( $pid ) || ! $pid ) {
            return null;
        }

        // set contact email meta
        if ( $contact_email ) {
            update_post_meta( $pid, 'rccc_contact_email', sanitize_email( $contact_email ) );
        } else {
            $user = get_userdata( $user_id );
            if ( $user && ! empty( $user->user_email ) ) {
                update_post_meta( $pid, 'rccc_contact_email', sanitize_email( $user->user_email ) );
            }
        }

        // Optionally set other minimal meta so the contractor list will show something
        // update_post_meta( $pid, 'rccc_city', '' );
        // update_post_meta( $pid, 'rccc_zips_csv', '' );

        return get_post( $pid );
    }
}

/* --- ACF save_post hook: ensure contractor post_author is current user and meta contact email set --- */
add_action( 'acf/save_post', 'rccc_acf_ensure_contractor_ownership', 20 );
function rccc_acf_ensure_contractor_ownership( $post_id ) {
    // only run for contractor post type
    if ( get_post_type( $post_id ) !== 'contractor' ) {
        return;
    }

    // require a logged-in user
    if ( ! is_user_logged_in() ) {
        return;
    }

    $current_user_id = get_current_user_id();
    if ( ! $current_user_id ) return;

    // If post_author is not current user, update it
    $post = get_post( $post_id );
    if ( $post && (int) $post->post_author !== (int) $current_user_id ) {
        // Only change author if appropriate
        wp_update_post( array( 'ID' => $post_id, 'post_author' => $current_user_id ) );
    }

    // Ensure rccc_contact_email meta is set to the user's email (if empty)
    $meta_email = get_post_meta( $post_id, 'rccc_contact_email', true );
    if ( empty( $meta_email ) ) {
        $user = wp_get_current_user();
        if ( $user && ! empty( $user->user_email ) ) {
            update_post_meta( $post_id, 'rccc_contact_email', sanitize_email( $user->user_email ) );
        }
    }
}

/* --- Logout shortcode and floating logout button --- */
/* Shortcode: [rccc_logout] */
add_shortcode( 'rccc_logout', function( $atts ) {
    if ( ! is_user_logged_in() ) return '';
    $redirect = isset( $atts['redirect'] ) ? esc_url( $atts['redirect'] ) : home_url( '/' );
    $url = wp_logout_url( $redirect );
    return '<a class="rccc-logout-link" href="' . esc_url( $url ) . '" style="display:inline-block;background:#047857;color:#fff;padding:8px 12px;border-radius:999px;text-decoration:none;">Log out</a>';
} );

/* Floating logout button visible to logged-in users (can be disabled by filter 'rccc_show_floating_logout') */
add_action( 'wp_footer', 'rccc_print_floating_logout_button' );
function rccc_print_floating_logout_button() {
    if ( ! is_user_logged_in() ) return;
    if ( apply_filters( 'rccc_show_floating_logout', true ) === false ) return;

    $logout_url = wp_logout_url( home_url( '/' ) );
    echo '<a aria-label="Log out" id="rccc-floating-logout" href="' . esc_url( $logout_url ) . '" style="
        position:fixed;
        right:16px;
        bottom:16px;
        background:#047857;
        color:#ffffff;
        padding:10px 14px;
        border-radius:999px;
        z-index:99999;
        text-decoration:none;
        box-shadow:0 6px 18px rgba(0,0,0,0.12);
        font-weight:600;
    ">Log out</a>';
}