<?php
/**
 * Plugin Name: RCCC Rematch & Log
 * Description: Admin tool to rematch leads (projects) to contractors and resend notifications. Also can auto-rematch when contractors are saved (opt-in).
 * Version: 1.0.0
 * Author: RCCC
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/* === CONFIG: set to true to auto-rematch projects when a contractor is published/updated === */
if ( ! defined( 'RCCC_AUTO_REMATCH_ON_CONTRACTOR_SAVE' ) ) define( 'RCCC_AUTO_REMATCH_ON_CONTRACTOR_SAVE', false );

/* admin page */
add_action( 'admin_menu', function() {
    add_management_page( 'RCCC Rematch & Log', 'RCCC Rematch & Log', 'manage_options', 'rccc-rematch-log', 'rccc_rematch_log_page' );
} );

function rccc_rematch_log_page() {
    if ( ! current_user_can( 'manage_options' ) ) wp_die( 'Permission denied' );

    $msg = '';
    if ( isset( $_POST['rccc_action'] ) && check_admin_referer( 'rccc_rematch_nonce', 'rccc_rematch_nonce_field' ) ) {
        $action = sanitize_text_field( $_POST['rccc_action'] );
        if ( $action === 'rematch_project' ) {
            $project_id = intval( $_POST['rccc_project_id'] ?? 0 );
            $do_send = ! empty( $_POST['rccc_send'] );
            if ( $project_id > 0 ) {
                $result = rccc_rematch_project_and_notify( $project_id, $do_send );
                $msg = '<div class="notice notice-success"><p>Rematch result: ' . esc_html( $result ) . '</p></div>';
            } else {
                $msg = '<div class="notice notice-error"><p>Enter a valid Project ID.</p></div>';
            }
        } elseif ( $action === 'rematch_all' ) {
            $do_send = ! empty( $_POST['rccc_send_all'] );
            $limit = intval( $_POST['rccc_limit'] ?? 100 );
            $count = rccc_rematch_all_projects( $do_send, $limit );
            $msg = '<div class="notice notice-success"><p>Processed ' . intval( $count ) . ' projects (limit ' . intval( $limit ) . '). Check debug log for details.</p></div>';
        }
    }

    echo '<div class="wrap"><h1>RCCC Rematch & Log</h1>';
    echo $msg;
    echo '<form method="post">';
    wp_nonce_field( 'rccc_rematch_nonce', 'rccc_rematch_nonce_field' );
    echo '<table class="form-table"><tr><th>Project ID</th><td><input type="number" name="rccc_project_id" class="regular-text" required></td></tr>';
    echo '<tr><th>Send emails?</th><td><label><input type="checkbox" name="rccc_send" value="1"> Send emails to matched contractors / homeowner</label></td></tr></table>';
    submit_button( 'Rematch Project', 'primary', 'rccc_action' );
    echo '<input type="hidden" name="rccc_action" value="rematch_project" />';
    echo '</form>';

    echo '<hr><h2>Rematch multiple projects</h2>';
    echo '<form method="post">';
    wp_nonce_field( 'rccc_rematch_nonce', 'rccc_rematch_nonce_field' );
    echo '<table class="form-table"><tr><th>Limit</th><td><input type="number" name="rccc_limit" value="100" class="small-text"> (max projects to process)</td></tr>';
    echo '<tr><th>Send emails?</th><td><label><input type="checkbox" name="rccc_send_all" value="1"> Send emails during processing</label></td></tr></table>';
    submit_button( 'Rematch Published Projects', 'secondary', 'rccc_action' );
    echo '<input type="hidden" name="rccc_action" value="rematch_all" />';
    echo '</form>';

    echo '<p>This tool logs actions to wp-content/debug.log (enable WP_DEBUG & WP_DEBUG_LOG for full details).</p>';
    echo '</div>';
}

/* core rematch runner - returns human readable result */
function rccc_rematch_project_and_notify( $project_id, $do_send = false ) {
    $project = get_post( $project_id );
    if ( ! $project ) return "Project {$project_id} not found.";

    // meta keys used in your site based on prior debugging
    $zip = get_post_meta( $project_id, 'rccc_zip', true );
    $category = get_post_meta( $project_id, 'rccc_category', true );
    $zip_clean = preg_replace( '/[^0-9]/', '', (string) $zip );
    $cat_name = trim( (string) $category );

    error_log("[RCCC REMATCH] Starting rematch for Project {$project_id} zip='{$zip_clean}' category='{$cat_name}'");

    // find term if possible
    $term = null;
    if ( $cat_name !== '' ) {
        $term = get_term_by( 'name', $cat_name, 'service_category' ) ?: get_term_by( 'slug', sanitize_title( $cat_name ), 'service_category' );
        if ( $term ) {
            error_log("[RCCC REMATCH] Found term id={$term->term_id} name={$term->name}");
        } else {
            error_log("[RCCC REMATCH] No term found for '{$cat_name}'");
        }
    }

    // build contractor query (same logic your debug used)
    $args = [
        'post_type' => 'contractor',
        'post_status' => 'publish',
        'posts_per_page' => -1,
        'fields' => 'ids',
        'meta_query' => [],
    ];
    if ( $zip_clean !== '' ) {
        $args['meta_query'][] = [ 'key' => 'rccc_zips_csv', 'value' => $zip_clean, 'compare' => 'LIKE' ];
    }
    if ( $term ) {
        $args['tax_query'] = [[ 'taxonomy' => 'service_category', 'field' => 'term_id', 'terms' => [ intval( $term->term_id ) ] ]];
    }

    error_log("[RCCC REMATCH] Query args: " . print_r( $args, true ) );
    $found = get_posts( $args );
    $count = is_array( $found ) ? count( $found ) : 0;
    error_log("[RCCC REMATCH] Found {$count} contractors for project {$project_id}");

    // record matched ids meta for admin visibility
    update_post_meta( $project_id, 'rccc_matched_ids', $found ? implode( ',', $found ) : '' );

    if ( $count === 0 ) {
        // Optionally notify admin that no matches found (keeps behavior consistent)
        $admin_email = get_option( 'admin_email' );
        if ( is_email( $admin_email ) ) {
            $subject = 'RCCC: No contractors matched this lead (rematch)';
            $body = "Project ID: {$project_id}\nCategory: {$cat_name}\nZIP: {$zip_clean}\nRematch run by " . wp_get_current_user()->user_login;
            wp_mail( $admin_email, $subject, $body );
        }
        return "No contractors found for project {$project_id}.";
    }

    // Send notifications (if enabled)
    $sent_count = 0;
    foreach ( $found as $cid ) {
        $c_email = get_post_meta( $cid, 'rccc_contact_email', true );
        if ( is_email( $c_email ) ) {
            // avoid duplicate sends per project->contractor
            $sent_flag = get_post_meta( $project_id, 'rccc_notified_to_' . $cid, true );
            if ( $sent_flag ) {
                error_log("[RCCC REMATCH] Already notified contractor {$cid} for project {$project_id}; skipping");
            } else {
                if ( $do_send ) {
                    $subject = 'New lead matching your services';
                    $body = "A lead matches your services.\nProject ID: {$project_id}\nCategory: {$cat_name}\nZIP: {$zip_clean}\nProject link: " . get_permalink( $project_id );
                    $headers = [ 'From: ' . get_bloginfo( 'name' ) . ' <no-reply@' . preg_replace( '/^www\\./', '', $_SERVER['HTTP_HOST'] ) . '>' ];
                    $sent = wp_mail( $c_email, $subject, $body, $headers );
                    error_log("[RCCC REMATCH] Email to {$c_email} for project {$project_id}: " . ( $sent ? 'OK' : 'FAILED' ) );
                }
                update_post_meta( $project_id, 'rccc_notified_to_' . $cid, current_time( 'mysql' ) );
                $sent_count++;
            }
        } else {
            error_log("[RCCC REMATCH] Contractor {$cid} has no valid rccc_contact_email");
        }
    }

    // homeowner email
    $home_email = get_post_meta( $project_id, 'rccc_homeowner_email', true );
    if ( $do_send && is_email( $home_email ) ) {
        $hsub = 'Your lead has been distributed to contractors';
        $hbody = "We found {$count} contractors for your project (ID {$project_id}).";
        $hheaders = [ 'From: ' . get_bloginfo( 'name' ) . ' <no-reply@' . preg_replace( '/^www\\./', '', $_SERVER['HTTP_HOST'] ) . '>' ];
        $hsent = wp_mail( $home_email, $hsub, $hbody, $hheaders );
        error_log("[RCCC REMATCH] Homeowner email to {$home_email}: " . ( $hsent ? 'OK' : 'FAILED' ) );
    } elseif ( ! $do_send ) {
        error_log("[RCCC REMATCH] Dry run - not sending homeowner email (home_email: {$home_email})");
    }

    return "Found {$count} contractors; rematch processed. Notifications sent to {$sent_count} contractors (if send enabled).";
}

/* process multiple published projects */
function rccc_rematch_all_projects( $do_send = false, $limit = 100 ) {
    $q = new WP_Query( [
        'post_type' => 'project',
        'post_status' => 'publish',
        'posts_per_page' => $limit,
        'fields' => 'ids',
    ] );
    $count = 0;
    if ( $q->have_posts() ) {
        foreach ( $q->posts as $pid ) {
            rccc_rematch_project_and_notify( $pid, $do_send );
            $count++;
        }
    }
    wp_reset_postdata();
    return $count;
}

/* Optional: auto-rematch when contractor saved/published (disabled by default) */
if ( RCCC_AUTO_REMATCH_ON_CONTRACTOR_SAVE ) {
    add_action( 'save_post_contractor', function( $post_id, $post, $update ) {
        if ( wp_is_post_revision( $post_id ) ) return;
        // Gather contractor zips and service terms
        $zips = get_post_meta( $post_id, 'rccc_zips_csv', true );
        $zip_items = array_filter( array_map( 'trim', explode( ',', (string) $zips ) ) );
        // For each zip, find projects matching and rematch
        foreach ( $zip_items as $zip ) {
            // find projects with that zip
            $project_ids = get_posts( [
                'post_type' => 'project',
                'post_status' => 'publish',
                'meta_query' => [[ 'key' => 'rccc_zip', 'value' => preg_replace('/[^0-9]/','',$zip), 'compare' => 'LIKE' ]],
                'fields' => 'ids',
                'posts_per_page' => -1,
            ] );
            foreach ( $project_ids as $pid ) {
                rccc_rematch_project_and_notify( $pid, true );
            }
        }
    }, 20, 3 );
}